package ru.yandex.mail.promocode.dao;

import java.sql.Types;
import java.util.List;
import java.util.Optional;

import javax.inject.Named;
import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.SingleColumnRowMapper;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;
import org.springframework.stereotype.Repository;

/**
 * @author Sergey Galyamichev
 */
@Repository
public class PromoCodeRepository {
    private static final Logger LOG = LoggerFactory.getLogger(PromoCodeRepository.class);
    private NamedParameterJdbcTemplate jdbcTemplate;

    @Autowired
    public PromoCodeRepository(@Named("psqlDataSource") DataSource psqlDataSource) {
        this.jdbcTemplate = new NamedParameterJdbcTemplate(psqlDataSource);
    }

    public void savePromoCodes(String tag, List<String> codes) {
        LOG.info("savePromoCodes({}, {})", tag, codes.size());
        SqlParameterSource[] params = codes.stream()
                .map(code -> new MapSqlParameterSource().addValue("code", code)
                        .addValue("tag", tag))
                .toArray(SqlParameterSource[]::new);
        String sql =
                "INSERT INTO " +
                        "promocodes(tag, code, created_dt) " +
                "VALUES " +
                        "(:tag, :code, NOW()::timestamp)";
        jdbcTemplate.batchUpdate(sql, params);
        LOG.info("savePromoCodes(...) - ok");
    }

    public Optional<String> assignPromoCode(String tag, Long uid, String deviceId) {
        LOG.info("assignPromoCode({}, {}, {})", tag, uid, deviceId);
        SqlParameterSource params = new MapSqlParameterSource()
                .addValue("uid", uid, Types.DECIMAL)
                .addValue("deviceId", deviceId, Types.VARCHAR)
                .addValue("tag", tag, Types.VARCHAR);
        String sql =
                "UPDATE " +
                        "promocodes " +
                "SET " +
                        "uid = :uid, " +
                        "device_id = :deviceId, " +
                        "assigned_dt = NOW()::timestamp " +
                "WHERE code = (" +
                        "SELECT " +
                                "code " +
                        "FROM " +
                                "promocodes " +
                        "WHERE " +
                                "(uid IS NULL AND device_id IS NULL OR uid = :uid AND device_id = :deviceId) AND tag = :tag " +
                        "LIMIT " +
                                "1 " +
                ") RETURNING " +
                        "code";

        List<String> updated = jdbcTemplate.query(sql, params, SingleColumnRowMapper.newInstance(String.class));
        if (updated.size() > 1) {
            LOG.error("More than one promo code is used for uid {} and devive {}!", uid, deviceId);
        }
        return updated.stream()
                .findAny();

    }

    public List<Long> checkPromoCode(Long uid, String deviceId, String code) {
        LOG.debug("findPromoCode({}, {}, {})", uid, deviceId, sanitize(code));
        MapSqlParameterSource params = new MapSqlParameterSource()
                .addValue("uid", uid, Types.DECIMAL)
                .addValue("code", code, Types.VARCHAR);
        if (deviceId != null) {
            params.addValue("deviceId", deviceId, Types.VARCHAR);
        }
        String sql =
                "SELECT " +
                        "id " +
                "FROM " +
                        "promocodes " +
                "WHERE " +
                        "uid = :uid AND " + (deviceId == null ? "device_id IS NULL" : "device_id = :deviceId") + " AND code = :code";
        List<Long> result = jdbcTemplate.query(sql, params, SingleColumnRowMapper.newInstance(Long.class));
        LOG.debug("findPromoCode(...) - found {}", result.size());
        return result;
    }

    public List<String> findPromoCode(String tag, Long uid, String deviceId) {
        LOG.debug("findPromoCode({}, {}, {})", tag, uid, deviceId);
        SqlParameterSource params = new MapSqlParameterSource()
                .addValue("uid", uid, Types.DECIMAL)
                .addValue("deviceId", deviceId, Types.VARCHAR)
                .addValue("tag", tag, Types.VARCHAR);
        String sql =
                "SELECT " +
                        "code " +
                "FROM " +
                        "promocodes " +
                 "WHERE " +
                        "(uid = :uid OR device_id = :deviceId) AND tag = :tag";
        List<String> result = jdbcTemplate.query(sql, params, SingleColumnRowMapper.newInstance(String.class));
        LOG.debug("findPromoCode(...) - found {}", result.size());
        return result;
    }

    private String sanitize(String code) {
        return code.length() > 3 ? code.substring(0, 2) + "***" : "***";
    }
}
