package ru.yandex.mail.promocode.service;

import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.ws.rs.BadRequestException;
import javax.ws.rs.NotFoundException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.stereotype.Service;

import ru.yandex.mail.promocode.PromoCodeService;
import ru.yandex.mail.promocode.dao.PromoCodeRepository;

/**
 * @author Sergey Galyamichev
 */
@Service("promoCodeService")
public class PromoCodesServiceImpl implements PromoCodeService {
    private static final Logger LOG = LoggerFactory.getLogger(PromoCodesServiceImpl.class);

    private final PromoCodeRepository promoCodeRepository;

    public PromoCodesServiceImpl(PromoCodeRepository promoCodeRepository) {
        this.promoCodeRepository = promoCodeRepository;
    }

    @Override
    public void upload(String tag, @Nonnull List<String> promoCodes) {
        if (tag == null) {
            throw new BadRequestException("Required non null parameter tag!");
        }
        try {
            promoCodeRepository.savePromoCodes(tag, promoCodes);
        } catch (DuplicateKeyException e) { //managed by resources/db/migration/V1.0__init.sql/unq_tag_code'
            throw new BadRequestException(e.getCause().getMessage());
        }
    }

    @Override
    public String assign(String tag, Long uid, @Nullable String deviceId) {
        if (tag == null || uid == null) {
            throw new BadRequestException("Required non null parameters tag and uid!");
        }
        List<String> promoCodes = promoCodeRepository.findPromoCode(tag, uid, deviceId);
        if (promoCodes.isEmpty()) {
            try {
                return promoCodeRepository.assignPromoCode(tag, uid, deviceId)
                        .orElseThrow(NotFoundException::new);
            } catch (DuplicateKeyException e) {
                //managed by resources/db/migration/V1.0__init.sql/unq_tag_uid_device_id'
                List<String> codes = promoCodeRepository.findPromoCode(tag, uid, deviceId);
                if (codes.size() != 1) {
                    LOG.error("Something strange...", e);
                }
                return codes.stream().findAny()
                        .orElseThrow(NotFoundException::new);
            }
        }
        return promoCodes.stream().findAny()
                .orElseThrow(NotFoundException::new);
    }

    @Override
    public void validate(Long uid, String deviceId, String promoCode) {
        if (promoCode == null || uid == null) {
            throw new BadRequestException("Required non null parameters uid and promoCode!");
        }
        List<Long> result = promoCodeRepository.checkPromoCode(uid, deviceId, promoCode);
        if (result.size() == 0){
            throw new NotFoundException("User with uid = " + uid +
                    " and deviceId = " + deviceId + " isn't related to the promocode");
        }
    }
}
