import click
import os
import yaml
import asyncio
import ujson

from typing import Mapping, Union
from datetime import timedelta
from pathlib import Path
from aiohttp import web
from aiohttp.web_response import Response
from yaml import SafeLoader, MappingNode

from mail.python.theatre.app.settings import LogSettings
from mail.python.theatre.app.log_helpers.app_logs import init_app_logging

from mail.puli.lib.dbstat.dbstat import make_db_stat_director


def init_logging(config: Mapping[str, Union[str, Mapping]]) -> None:
    log_config = config['log']

    log = LogSettings()
    log.dir_path = Path(log_config['dir'])
    log.level = log_config['level']
    init_app_logging(settings=log, env=None)


async def create_app(config: Mapping[str, Union[str, Mapping]]) -> web.Application:
    import uvloop
    asyncio.set_event_loop_policy(uvloop.EventLoopPolicy())

    app = web.Application(loop=asyncio.get_event_loop())

    dbstat = make_db_stat_director(config)

    async def unistat_handler(_):
        return Response(text=ujson.dumps(dbstat.stats), content_type='application/json')

    async def shutdown(_):
        await dbstat.stop()

    await dbstat.start()

    app.router.add_get('/unistat', unistat_handler)
    app.on_shutdown.append(shutdown)

    return app


@click.command()
@click.option('--host', default='::0', help='IP addr to bind to')
@click.option('--port', default=80, help='HTTP port to listen')
@click.option('--dir', 'working_dir', default='.', help='Working directory')
@click.option('--config', 'config_file', help='Configuration file')
def main(host, port, working_dir, config_file):
    os.chdir(working_dir)

    def timedelta_yaml_constructor(loader: SafeLoader, node: MappingNode) -> timedelta:
        return timedelta(**loader.construct_mapping(node))

    yaml.add_constructor('!TimeDelta', timedelta_yaml_constructor, Loader=yaml.SafeLoader)

    with open(config_file, 'r') as f:
        config = yaml.safe_load(f)

    init_logging(config)

    web.run_app(
        create_app(config),
        port=port,
        host=host,
        access_log_format='%a %t "%r" %s %b "%{Referer}i" "%{User-Agent}i" %Tf'
    )


if __name__ == '__main__':
    main()
