import click
import os
import sys
import yaml
import logging
import asyncio

from typing import Mapping, Union
from datetime import timedelta
from yaml import SafeLoader, MappingNode

from logging.handlers import WatchedFileHandler

from mail.puli.lib.dbtask.base_task import create_task


def init_logging(config: Mapping[str, Union[str, Mapping]], task_name: str) -> None:
    log_config = config['log']

    log_dir = log_config['dir']
    log_level = log_config['level']

    if task_name:
        os.makedirs(log_dir, mode=0o755, exist_ok=True)
        log_file = os.path.join(log_dir, '{}.log'.format(task_name))
        handler = WatchedFileHandler(log_file)
    else:
        handler = logging.StreamHandler(sys.stdout)

    formatter = logging.Formatter('%(asctime)s [%(levelname)s] %(message)s')
    handler.setFormatter(formatter)

    root_logger = logging.getLogger()
    root_logger.addHandler(handler)
    root_logger.setLevel(logging.getLevelName(log_level.upper()))


@click.command()
@click.option('--datasource', help='Datasource to connect to')
@click.option('--task', 'task_name', help='Task name to execute')
@click.option('--dir', 'working_dir', default='.', help='Working directory')
@click.option('--config', 'config_file', help='Configuration file')
def main(datasource, task_name, working_dir, config_file):
    os.chdir(working_dir)

    def timedelta_yaml_constructor(loader: SafeLoader, node: MappingNode) -> timedelta:
        return timedelta(**loader.construct_mapping(node))

    yaml.add_constructor('!TimeDelta', timedelta_yaml_constructor, Loader=yaml.SafeLoader)

    with open(config_file, 'r') as f:
        config = yaml.safe_load(f)

    task = create_task(task_name, config)

    init_logging(config, task.name)

    asyncio.get_event_loop().run_until_complete(task.run(datasource))


if __name__ == '__main__':
    main()
