from typing import Sequence, Mapping, Union
from itertools import chain
from datetime import timedelta

from mail.pypg.pypg.query_conf import load, QueriesHolder

from mail.python.theatre.roles import Director
from mail.python.theatre.stages.db_stats.types import DbSignal
from mail.python.theatre.stages.db_stats.roles.db_stat_poller import DbStatPoller
from mail.python.theatre.profiling.typing import Metrics

from .dbsources import make_db_sources


class DbStatDirector(Director):
    def __init__(self, db_stat_pollers: Sequence[DbStatPoller]):
        self.db_stat_pollers = db_stat_pollers
        super().__init__(tasks=self.db_stat_pollers)

    @property
    def stats(self) -> Metrics:
        return list(chain.from_iterable(poller.stats for poller in self.db_stat_pollers))


def load_query_conf(filename: str) -> QueriesHolder:
    with open(filename) as fd:
        return load(fd.read().split('\n'))


def make_db_stat_director(config: Mapping[str, Union[str, Mapping]]) -> DbStatDirector:
    dbqueries = load_query_conf(config['dbsignals']['query_conf'])
    dbsources = make_db_sources(config)

    def to_timedelta(value: Union[int, float, timedelta]) -> timedelta:
        if isinstance(value, timedelta):
            return value
        elif isinstance(value, int) or isinstance(value, float):
            return timedelta(seconds=value)

        raise TypeError('invalid period value type: {}'.format(value))

    def name_formatter(name: str, value: str = None) -> str:
        if value:
            return f'{name}_{value}'
        return name

    db_stat_pollers = [
        DbStatPoller(
            db=dbsources[source],
            db_signal=DbSignal(
                query=getattr(dbqueries, signal['name']).query,
                run_every=to_timedelta(signal['period']),
                row_signal=signal.get('row_signal', False),
                sigopt_suffix=signal.get('sigopt_suffix', 'max'),
            ),
            signal_name_formatter=name_formatter
        )
        for source, signals in config['dbstats'].items()
        for signal in signals
        if source in dbsources
    ]

    return DbStatDirector(db_stat_pollers=db_stat_pollers)
