
import aiopg
import psycopg2
import logging

from typing import Mapping, Union
from aiopg import Connection


class BaseTask(object):
    name = None

    def __init__(self, config: Mapping[str, Union[str, Mapping]]):
        self.config = config
        self.logger = logging.getLogger(self.name)

    async def run(self, datasource: str) -> None:
        try:
            async with await self._get_db_connection(datasource) as connection:
                return await self._run(connection)
        except psycopg2.Warning as ex:
            self.logger.warning('PG_WARNING: {}'.format(ex))
        except psycopg2.Error as ex:
            self.logger.error('PG_ERROR: {}'.format(ex))
        except Exception as ex:
            self.logger.error('ERROR: {}: {}'.format(type(ex).__name__, ex))

    async def _run(self, _) -> None:
        raise NotImplementedError('task not implemented')

    async def _get_db_connection(self, datasource: str) -> Connection:
        ds = self.config['datasources'][datasource]
        return await aiopg.connect(
            host=ds['host'],
            port=ds['port'],
            database=ds['dbname'],
            user=ds['user']
        )

    def vars(self):
        return self.config['vars'][self.name]


def create_task(name, config: Mapping[str, Union[str, Mapping]]) -> BaseTask:
    tasks = {
        task.name: task for task in BaseTask.__subclasses__()
    }

    return tasks.get(name, BaseTask)(config)
