from mail.pypg.pypg.reflected import async_connect, reflect_db, begin, commit, rollback, is_locked
from contextlib import closing
from collections import namedtuple
from hamcrest.core.base_matcher import BaseMatcher
from time import sleep


ConcurrentCallResult = namedtuple('ConcurrentCallResult', ['value', 'n_locks'])

FunctionCallResult = namedtuple('FunctionCallResult', ['future', 'complete'])


class ConcurrentCall(object):
    ''' Invokes callable object concurrently and stores result of the call.
    Objects of this class are dedicated to be used with
   mail.pypg.hamcrest_support.concurrent.IsSerialized matcher. The matcher
    modifies the internal state of the class's object.
    '''

    def __init__(self, func):
        '''Creates object which stores given callable object to call it concurrently.

        Parameters
        ----------
        func : callable
            callable object which accepts database reflection object as an argument
            and returns a mail.pypg.pypg.reflected.Future object
            E.g.:
                def my_func(db):
                    return db.code.my_sql_func(i_arg1=some_value)
        '''

        self.timeout = 1
        self.func = func
        self.dsn = None
        self.delay = None
        self._result = None
        self._str = None

    def __connect(self):
        return async_connect(self.dsn)

    def __func(self, db):
        r = self.func(db)
        if self.delay:
            sleep(self.delay)
        return FunctionCallResult(r, get_completion(self.func))

    def execute(self):
        '''Invokes stored callable object within concurrent transactions and saves result as
       mail.pypg.hamcrest_support.concurrency.ConcurrentCallResult class object, which
        combines function call results with number of transaction locks have been occurred.

        Currently only two concurrent transactions will be executed. Thus two callable's
        invocation would be made.
        '''

        with closing(self.__connect()) as conn1, closing(self.__connect()) as conn2:
            conns = [conn1, conn2]
            dbs = [reflect_db(c) for c in conns]

            list(map(begin, conns))

            res = list(map(self.__func, dbs))

            locked = list(map(is_locked, conns))

            n_locks = sum(locked)

            assert n_locks < len(conns), "all transactions are locked - at least one transaction should not be locked"

            unlock_order = [0, 1] if locked[1] else [1, 0]

            for i in unlock_order:
                if not res[i].future.wait(self.timeout).has_exception():
                    res[i].complete(conns[i])

        self._result = ConcurrentCallResult((r.future.get() for r in res), n_locks)

    def __str__(self):
        return self._str if self._str is not None else "{}".format(self._result)

    def result(self):
        '''Provides result of a recent execute() call. If execute() has not been called and
        no result stored inside the object, execute() method will be invoked automatically.

        Returns
        -------
       mail.pypg.hamcrest_support.concurrency.ConcurrentCallResult
            a result of recent execute() call.
        '''

        if self._result is None:
            self.execute()
        return self._result

    def with_timeout(self, timeout):
        '''Specify timeout for asynchronous calls

        Parameters
        ----------
        timeout : double
            timeout in seconds

        Returns
        -------
       mail.pypg.hamcrest_support.concurrency.ConcurrentCall
            self, to allow to use it in methods call-chain style.
        '''

        self.timeout = timeout
        return self

    def on(self, dsn):
        '''Specifies DSN to connect to

        Parameters
        ----------
        dsn : string
            DSN itself

        Returns
        -------
       mail.pypg.hamcrest_support.concurrency.ConcurrentCall
            self, to allow to use it in methods call-chain style.
        '''

        self.dsn = dsn
        return self

    def with_delay(self, delay):
        ''' Specifies the delay between func calls.
        The delay is needed if we want the specific order of locks to be made.

        Parameters
        ----------
        delay : float
            Number of seconds to wait

        Returns
        -------
       mail.pypg.hamcrest_support.concurrency.ConcurrentCall
            self, to allow to use it in methods call-chain style.
        '''

        self.delay = delay
        return self


def concurrent_call(func):
    '''Creates concurrent call object to use it with matcher and Hamcrest framework

    Parameters
    ----------
    func : callable
        callable object which accepts database reflection object as an argument
        and returns a mail.pypg.pypg.reflected.Future object.

    Example
    -------
    E.g. we want to check if 'code.purge_operations' would lock:

        from mail.pypg.hamcrest_support.concurrency import concurrent_call, is_serialized

        #...

        assert_that(
            concurrent_call(lambda d: d.code.purge_operations(i_count=10)).on(dsn),
            not_(is_serialized()),
            "concurrent function calls should not be serialized"
        )

    Returns
    -------
   mail.pypg.hamcrest_support.concurrency.ConcurrentCall
        object which represents function concurrent call.
    '''

    return ConcurrentCall(func)


class IsSerialized(BaseMatcher):
    def _matches(self, call):
        if type(call) is not ConcurrentCall:
            raise TypeError("ConcurrentCall expected")
        if call.result().n_locks > 0:
            call._str = "serialized"
            return True
        else:
            call._str = "not serialized"
            return False

    def describe_to(self, description):
        description.append_text('serialized')


def is_serialized():
    '''Create serialized concurrent call matcher. Matches if calls
    have been serialized via transaction locks. Should be used with
   mail.pypg.hamcrest_support.concurrency.concurrent_call().

    Example
    -------
    E.g. we want to check if 'code.purge_operations' would lock:

        from mail.pypg.hamcrest_support.concurrency import concurrent_call, is_serialized

        #...

        assert_that(
            concurrent_call(lambda d: d.code.purge_operations(i_count=10)).on(dsn),
            not_(is_serialized()),
            "concurrent function calls should not be serialized"
        )

    Returns
    -------
    hamcrest.core.base_matcher.BaseMatcher
        Hamcrest matcher interface implementation
    '''

    return IsSerialized()


def get_completion(func):
    ''' Get function completion if specialized.
        This gives us possibility to add `complete` method to func which will determine how to end transaction.
        Can be used to call ROLLBACK instead of COMMIT at the end.
    '''
    complete = getattr(func, 'complete', None)
    return complete if callable(complete) else commit


class SequenceCaller(object):
    def __init__(self, functions):
        self.functions = iter(functions)
        self.current_func = None

    def __call__(self, db):
        self.current_func = next(self.functions)
        return self.current_func(db)

    @property
    def complete(self):
        return get_completion(self.current_func)


def sequence_caller(functions):
    ''' Creates SequenceCaller object to use it with concurrent_call.
        This object will invoke functions in given sequence on each call.

    Parameters
    ----------
    functions : list of callable objects in order to call
        Callable objects accepts database reflection object as an argument
        and returns a mail.pypg.pypg.reflected.Future object

    Example
    -------
    E.g. we want to check if 'code.purge_operations' and 'code.add_operation' would lock:

        from mail.pypg.hamcrest_support.concurrency import concurrent_call, is_serialized, sequence_caller

        #...

        assert_that(
            concurrent_call(sequence_caller([
                lambda d: d.code.purge_operations(i_count=10),
                lambda d: d.code.add_operation(i_op_id=42)
            ])).on(dsn),
            not_(is_serialized()),
            "concurrent function calls should not be serialized"
        )

    Returns
    -------
   mail.pypg.hamcrest_support.concurrency.SequenceCaller
        object which represents function sequential caller.
    '''

    return SequenceCaller(functions)


class RollbackCaller(object):
    def __init__(self, func):
        self.func = func

    def __call__(self, db):
        return self.func(db)

    @property
    def complete(self):
        return rollback


def with_rollback(func):
    ''' Creates RollbackCaller object to use it with concurrent_call.
        This object will call ROLLBACK to complete transaction instead of COMMIT.

    Parameters
    ----------
    func : callable
        callable object which accepts database reflection object as an argument
        and returns a mail.pypg.pypg.reflected.Future object.

    Example
    -------
    E.g. we want to check if 'code.purge_operations' works after 'code.add_operation' with ROLLBACK:

        from mail.pypg.hamcrest_support.concurrency import concurrent_call, is_serialized, sequence_caller, with_rollback

        #...

        call = concurrent_call(sequence_caller([
            with_rollback(lambda d: d.code.add_operation(i_op_id=42)),
            lambda d: d.code.purge_operations(i_count=10)
        ])).on(dsn)
        assert_that(call, is_serialized(), "concurrent function calls should be serialized")
        assert_that(list(call.result().value)[1], equal_to(SUCCESS))

    Returns
    -------
   mail.pypg.hamcrest_support.concurrency.RollbackCaller
        object which represents function caller with ROLLBACK at the end.
    '''

    return RollbackCaller(func)
