# coding: utf-8

import backoff
import select
import logging

import psycopg2
import psycopg2.extensions as PGE

from mail.pypg.pypg import logged_connection


LOG_SQL = 25
log = logging.getLogger(__name__)

logging.addLevelName(LOG_SQL, "SQL")


TX_STATUS = {
    PGE.TRANSACTION_STATUS_IDLE:    'no',
    PGE.TRANSACTION_STATUS_ACTIVE:  'active',
    PGE.TRANSACTION_STATUS_INTRANS: 'intrans',
    # The session is idle in a failed transaction block.
    PGE.TRANSACTION_STATUS_INERROR: 'failed',
    # Reported if the connection with the server is bad.
    PGE.TRANSACTION_STATUS_UNKNOWN: 'bad',
}


def global_register_unicode():
    PGE.register_type(PGE.UNICODE)
    PGE.register_type(PGE.UNICODEARRAY)


class LoggingConnection(logged_connection.LoggingConnection):

    LOG_FMT = u"[pid:{pid}] {msg} (execution time: {ms} ms)"

    def desc(self):
        return \
            'Connection(pid:{pid}, tx:{tx}, closed:{closed})'.format(
                pid=self.get_backend_pid(),
                tx=TX_STATUS.get(self.get_transaction_status()),
                closed=self.closed,
            )

    def wait(self):
        if not self.async_:
            return
        self._print_notices()
        while 1:
            try:
                state = self.poll()
            finally:
                self._print_notices()
            if state == PGE.POLL_OK:
                break
            elif state == PGE.POLL_WRITE:
                select.select([], [self.fileno()], [])
            elif state == PGE.POLL_READ:
                select.select([self.fileno()], [], [])
            else:
                raise RuntimeError("poll() returned %s" % state)

    def wait_some(self, timeout):
        if not self.async_:
            return
        self._print_notices()
        state = self.poll()
        if state == PGE.POLL_OK:
            return
        if state == PGE.POLL_READ:
            select.select([self.fileno()], [], [], timeout)

    def close(self):
        self.wait()
        return super(LoggingConnection, self).close()


@backoff.on_exception(backoff.expo, psycopg2.OperationalError, max_tries=3)
def make_connection(*args, **kwargs):
    autocommit = kwargs.pop('autocommit', None)
    kwargs['connection_factory'] = LoggingConnection
    conn = psycopg2.connect(*args, **kwargs)
    if autocommit is not None:
        conn.autocommit = autocommit
    conn.wait()
    return conn


class Connector(object):
    def __init__(self, *args, **kwargs):
        self.args = args
        self.kwargs = kwargs

    def connect(self, **kwargs):
        conn_kwags = self.kwargs.copy()
        conn_kwags.update(kwargs)
        return make_connection(
            *self.args,
            **conn_kwags
        )


def refresh_connection(conn):
    conn.wait()
    cur = conn.cursor()
    if conn.get_transaction_status() != PGE.TRANSACTION_STATUS_IDLE:
        cur.execute("ROLLBACK")
        conn.wait()
        cur.close()
