# coding: utf-8

from datetime import datetime
from enum import Enum
from six import text_type, binary_type

from .types.adapted import BaseAdaptedComposite, BaseAdaptedList
from .types.db_enums import DBEnum

SEP = '\t'
NULL = '\\N'
BNULL = NULL.encode('utf-8')


def escape_datetime(val):
    return val.isoformat(' ').encode('utf-8')


def should_be_quouted(val):
    for char in b'(),"\\ \n\r\t':
        if char in val:
            return True
    return val == b''


def should_be_quouted_in_array(val):
    for char in b'{}':
        if char in val:
            return True
    return should_be_quouted(val) or (val == b'NULL')


def in_array(val):
    return (
        val
        .replace(b'\\', b'\\\\')
        .replace(b'"', b'\\"')
    )


def escape_composite_value(val):
    # http://www.postgresql.org/docs/9.4/interactive/rowtypes.html#ROWTYPES-IO-SYNTAX
    if val is None:
        # A completely empty field value
        # represents a NULL.
        return b''
    if isinstance(val, Enum):
        val = val.value
    if isinstance(val, text_type):
        val = val.encode('utf-8')
    elif isinstance(val, binary_type):
        val = val
    elif isinstance(val, datetime):
        val = escape_datetime(val)
    else:
        val = text_type(val).encode('utf-8')

    # Double quotes and backslashes embedded in field values will be doubled.
    val = val.replace(b'"', b'""').replace(b'\\', b'\\\\')

    # Put double quotes around field values
    # if they are empty strings or contain parentheses, commas,
    # double quotes, backslashes, or white space.
    if should_be_quouted(val):
        val = b'"%s"' % val

    return val


def isseq(s):
    import collections.abc
    from inspect import isgenerator

    return (isinstance(s, collections.abc.Sequence) or isgenerator(s)) \
        and not isinstance(s, text_type) and not isinstance(s, binary_type)


def escape_composite(val):
    return b'(%s)' % b','.join(
        escape_composite_value(getattr(val, f)) for f in val.fields()
    )


def escape_seq(seq):
    def quote(v):
        return v if not should_be_quouted_in_array(v) else b'"%s"' % v
    if any(v is None for v in seq):
        raise NotImplementedError('Unhandled NULL in array values %r', seq)
    return b'{%s}' % b','.join(quote(in_array(escape(v))) for v in seq)

ESCAPE_PLAN = [
    (binary_type, lambda val: val),
    (BaseAdaptedList, lambda val: escape_seq(val.seq)),
    (BaseAdaptedComposite, escape_composite),
    (text_type, lambda val: val.encode('utf-8')),
    (DBEnum, lambda val: val.value.encode('utf-8')),
    (datetime, escape_datetime),
]


def escape(val):
    if val is None:
        return BNULL
    if isseq(val):
        return escape_seq(val)
    for obj_type, escaper in ESCAPE_PLAN:
        if isinstance(val, obj_type):
            return escaper(val)
    return text_type(val).encode('utf-8')


def pgcopy(val):
    val = escape(val)
    # The specified null string is sent by COPY TO without adding any backslashes
    if val != BNULL:
        val = (
            val
            .replace(b'\\', b'\\\\')
            .replace(b'\n', b'\\n')
            .replace(b'\r', b'\\r')
            .replace(b'\t', b'\\t')
        )
    return val
