from __future__ import print_function

import psycopg2.extras
import psycopg2.extensions
from abc import ABCMeta
from contextlib import closing
from collections import namedtuple

psycopg2.extras.register_uuid()


SqlFunctionArgument = namedtuple('SqlFunctionArgument', ['name', 'type'])


def ready(conn):
    """Displays if connection is ready for request

    Parameters
    ----------
    conn : psycopg2.connection
        connection to examine

    Returns
    -------
    bool
    """

    return not conn.isexecuting()


def wait(conn, timeout=None):
    """Wait until a connection has data available or timeout expired.
    If timeout expired the active request will be cancelled via conn.cancel()
    and the loop will be broken by a server error.

    Parameters
    ----------
    conn : psycopg2.connection
        connection in asynchronous mode
    timeout : double, optional
        timeout to wait (the default is None, which means infinity)

    Raises
    ------
    psycopg2.connection.OperationalError
        in case of bad connection poll status

    """

    assert conn.async_, "connection should be in asynchronous mode"

    if ready(conn):
        return

    import select
    from psycopg2.extensions import POLL_OK, POLL_READ, POLL_WRITE

    def cancel_on_timeout(fds):
        if len(fds) == 0:
            conn.cancel()

    while 1:
        try:
            state = conn.poll()
            if state == POLL_OK:
                break
            elif state == POLL_READ:
                r, _, _ = select.select([conn.fileno()], [], [], timeout)
                cancel_on_timeout(r)
            elif state == POLL_WRITE:
                _, w, _ = select.select([], [conn.fileno()], [], timeout)
                cancel_on_timeout(w)
            else:
                raise conn.OperationalError("bad state from poll: %s" % state)
        except KeyboardInterrupt:
            conn.cancel()
            continue


def _rows_to_function_result(rows, multi_row=True):
    result = [row[0] for row in rows]
    if multi_row:
        return result
    else:
        assert len(result) == 1
        return result[0]


class Future(object):
    '''Future concept implementation. Used as a result of asynchronous operation
    with a database. Beware of timeouts from DB side because the IO loop for receiving
    result will be called only at blocking methods like `wait()` or `get()`.
    '''

    __metaclass__ = ABCMeta

    def __init__(self, cursor, multi_row=True):
        '''Construct future object for an asynchronous IO has been initiated by a request on a given cursor.

        Parameters
        ----------
        cursor : psycopg2.cursor
            Cursor on which the request has been made.
        multi_row : bool, optional
            Indicates if result should be treated as a single element instead of sequence of elements;
            if value is true and multiply or no rows are returned by request - an assertion will fail.
            (the default is `True`, which means - single row is expected and should be returned by request)
        '''

        assert cursor is not None, "cursor can not be None"
        self._cursor = cursor
        self._result = None
        self._multi_row = multi_row

    def wait(self, timeout=None):
        '''Wait for result being ready to read and read it. The operation would block
        until result being received or timeout expired.

        Parameters
        ----------
        timeout : double, optional
            timeout for IO in seconds (the default is None, which means infinity)

        Returns
        -------
       mail.pypg.pypg.reflected.Future
            self reference
        '''

        if not self.is_ready():
            assert self._result is None and self._cursor is not None
            try:
                with closing(self._cursor) as cur:
                    wait(cur.connection, timeout)
                    rows = cur.fetchall()
                    self._result = _rows_to_function_result(rows, self._multi_row)
            except Exception as e:
                self._result = e
            self._cursor = None
        return self

    def is_ready(self):
        '''Indicates if future is ready - result has been received.
        Method would not block.

        Returns
        -------
        bool
            `True` - if result has been received,
            `False` - otherwise.
        '''

        return self._cursor is None

    def has_exception(self):
        '''Indicates if the result contains an exception. Should be called only
        if a future is ready (is_ready() return true).

        Returns
        -------
        bool
            `True` - if exception has been raised or received from a database,
            `False` - otherwise.
        '''

        assert self.is_ready(), "Future must be in ready state to provide information about result"
        return type(self._result) is Exception

    def get(self):
        '''Return request result. Would block if a future is not ready.

        Raises
        ------
        Exception
            exception if it has been raised during result receive or received from a database.

        Returns
        -------
        depends-on-request
            result of request from a database: single object or sequence of objects, depends on
            multi_row constructor parameter.
        '''

        if self.wait().has_exception():
            raise self._result
        return self._result


class SqlFunction(object):
    '''Callable SQL function reflection. Would be used as an ordinary function. Has
    same parameters, their types and names as it's SQL origin. It automatically
    resolves the result type of call in terms of single row or multiple rows.

    Attributes
    ----------
    schema : str
        Schema name. E.g.: for full qualified function name 'code.add_operation' schema is 'code'.
    name : str
        Function name. E.g.: for full qualified function name 'code.add_operation' name is 'add_operation'.

    Example
    -------
    Given SQL function:

        REATE OR REPLACE FUNCTION code.add_operation (
            i_id            uuid,
            i_type          mops.operation_type,
            i_uid           bigint,
            i_data          jsonb,
            i_request_id    text)
        RETURNS code.add_operation_result AS $$

    Will be reflected as SqlFunction equivalent to the following pseudo-python code:

        def code.add_operation(i_id, i_type, i_uid, i_data, i_request_id):
            if cursor.connection.async_:
                # Will return future on the result with code.add_operation_result type
                return Future(cursor)
            #Returns result immediately
            return code.add_operation_result(cursor.fetch())
    '''

    __metaclass__ = ABCMeta

    def __init__(self, conn, decl):
        self._multi_row = decl.multi_row
        self.schema = decl.schema
        self.name = decl.name

        self._args_list = [SqlFunctionArgument(**a) for a in decl.args] if decl.args is not None else []
        self._conn = conn

    def __arg_names(self):
        return [k.name for k in self._args_list]

    def __check_args(self, query_args):
        arg_names = set(self.__arg_names())
        if set(query_args.keys()) != arg_names:
            unknown_args = set(query_args.keys()) - arg_names
            missed_args = arg_names - set(query_args.keys())
            raise ValueError('bad arguments provided; unknown: {unknown_args}, missed: {missed_args}'.format(
                unknown_args=', '.join((k for k in unknown_args)),
                missed_args=', '.join((k for k in missed_args))
            ))

    def __make_args_list(self, query_args):
        def format_arg(arg):
            return '%({name})s::{type}'.format(name=arg.name, type=arg.type)
        return (format_arg(k) for k in self._args_list)

    def __make_query(self, query_args):
        self.__check_args(query_args)
        return 'SELECT {schema}.{function}({args})'.format(
            schema=self.schema,
            function=self.name,
            args=', '.join(self.__make_args_list(query_args))
        )

    def __execute_query(self, cursor, query_args):
        cursor.execute(self.__make_query(query_args), vars=query_args)

    def __cursor(self):
        return self._conn.cursor(cursor_factory=psycopg2.extras.NamedTupleCursor)

    def __call(self, query_args):
        with closing(self.__cursor()) as cur:
            try:
                self.__execute_query(cur, query_args)
                return _rows_to_function_result(cur.fetchall(), self._multi_row)
            except:
                print(cur.query)
                raise

    def __async_call(self, query_args):
        cur = self.__cursor()
        self.__execute_query(cur, query_args)

        return Future(cur, self._multi_row)

    def __call__(self, *args, **kwargs):
        assert not self._conn.closed, "connection is closed"
        args = dict(zip(self.__arg_names(), args))
        args.update(kwargs)
        if self._conn.async_:
            return self.__async_call(args)
        return self.__call(args)


def register_composite(name, conn):
    '''Asynchronous connection compatible version of psycopg2.extras.register_composite
    Original version is not compatible with asynchronous connection thus to register
    composites for such connection is a problem. This function solves such problem.

    Parameters
    ----------
    name : str
        fully qualified name of composite
    conn : psycopg2.connection
        connection to register composite to

    Returns
    -------
    see-original-function
        result of `psycopg2.extras.register_composite(name, conn)` call
    '''

    if conn.async_:
        class ConnectionProxy(type(conn)):
            def __init__(self, impl):
                self._impl = impl

            def cursor(self, *args, **kwargs):
                class AsyncCursorProxy(psycopg2.extensions.cursor):

                    def fetchall(self):
                        wait(self.connection)
                        return super(AsyncCursorProxy, self).fetchall()

                kwargs['cursor_factory'] = AsyncCursorProxy
                return self._impl.cursor(*args, **kwargs)

        conn = ConnectionProxy(conn)

    return psycopg2.extras.register_composite(name, conn)


class Composite(object):
    '''Composite type reflection. Would be used to create an object which
    relates to the respective composite type in database. Composite type
    system is based on the original psycopg2 composite registration engine.
    Composite is callable, call creates object of the composite reflection
    class.

    Attributes
    ----------
    schema : str
        Schema name. E.g.: for full qualified composite name 'code.add_operation_result' schema is 'code'.
    name : str
        Composite name. E.g.: for full qualified composite name 'code.add_operation_result' name is 'add_operation_result'.
    type : class
        Class of the reflected composite

    Example
    -------
    Given SQL composite type:

        CREATE TYPE code.add_operation_result AS (
            inserted    boolean,
            operation   text
        );

    To create respective object type in python:

        res = db.code.add_operation_result(true, "some text stuff")
    '''

    __metaclass__ = ABCMeta

    def __init__(self, conn, decl):
        self.schema = decl.schema
        self.name = decl.name
        self.type = register_composite('{}.{}'.format(self.schema, self.name), conn).type

    def __call__(self, *args, **kwargs):
        return self.type(*args, **kwargs)


class Table(Composite):
    '''Table row type reflection. Would be used to create an object which relates to
    the respective table and composite type in database. All the tables in
    PostgreSQL have composite representations. What's why Table inherits Composite.

    Attributes
    ----------
    conn : psycopg2.connection
        Connection to make requests.
    '''

    def __init__(self, conn, decl):
        super(Table, self).__init__(conn, decl)
        self.conn = conn

    def select(self, *args, **kwargs):
        """Request simple query to select row from the table.

        Example
        -------
        E.g. we want to getall the rows with `uid` field equal to given uid value and
        ordered by `cid` field descending.

            res = db.mops.change_log.select('ORDER BY cid DESC', uid=uid)

        Parameters
        ----------
        *args : str
            sorting type
        **kwargs :
            conditions joined by 'AND'

        Returns
        -------
        namedtuple
            row of the table
        """

        condition = ''
        if len(kwargs) > 0:
            condition = 'WHERE {}'.format(' AND '.join(("{}='{}'".format(k, kwargs[k]) for k in kwargs)))
        sorting = 'ORDER BY 1'
        if len(args) > 0:
            sorting = ' '.join(args)
        query = '''
            SELECT *
            FROM {schema}.{table}
            {condition}
            {sorting}
        '''.format(
            schema=self.schema,
            table=self.name,
            condition=condition,
            sorting=sorting,
        )
        with closing(self.conn.cursor(cursor_factory=psycopg2.extras.NamedTupleCursor)) as cur:
            cur.execute(query)
            if self.conn.async_:
                wait(self.conn)
            result = cur.fetchall()
        return result

    def update(self, values, **kwargs):
        """Updates table entries with specified values.

        Example
        -------
        We want to update all the entries with specified `op_id` and `id` of `mops.message_chunks`
        with given `state`

            db.mops.message_chunks.update({'state': state}, op_id=op.id, id=1)

        Parameters
        ----------
        values : dict
            new entries values
        **kwargs :
            conditions joined by 'AND'

        """

        condition = ''
        if len(kwargs) > 0:
            condition = 'WHERE {}'.format('AND '.join(("{}='{}'".format(k, kwargs[k]) for k in kwargs)))
        query = '''
            UPDATE {schema}.{table} SET {values} {condition}
        '''.format(
            schema=self.schema,
            table=self.name,
            condition=condition,
            values=', '.join(("{k}=%({k})s".format(k=k) for k in values.keys())),
        )
        with closing(self.conn.cursor(cursor_factory=psycopg2.extras.NamedTupleCursor)) as cur:
            cur.execute(query, vars=values)
            if self.conn.async_:
                wait(self.conn)

    def delete(self, **kwargs):
        """Deletes table entries.

        Example
        -------
        We want to delete all the entries with specified `op_id` and `id` of `mops.message_chunks`

            db.mops.message_chunks.delete(op_id=op.id, id=1)

        Parameters
        ----------
        **kwargs :
            conditions joined by 'AND'

        """

        condition = ''
        if len(kwargs) > 0:
            condition = 'WHERE {}'.format('AND '.join(("{}='{}'".format(k, kwargs[k]) for k in kwargs)))
        query = '''
            DELETE FROM {schema}.{table} {condition}
        '''.format(
            schema=self.schema,
            table=self.name,
            condition=condition,
        )
        with closing(self.conn.cursor(cursor_factory=psycopg2.extras.NamedTupleCursor)) as cur:
            cur.execute(query)
            if self.conn.async_:
                wait(self.conn)


def connect(dsn, **kwargs):
    '''Make synchronous connection via `psycopg2.connect()`. Accepts all the arguments of
    the original function plus `autocommit` argument

    Parameters
    ----------
    dsn : str
        DSN of a database to connect to
    autocommit : bool
        Use autocommit in this session (the default is `True`)

    Returns
    -------
    psycopg2.connection
        connection established
    '''

    conn = psycopg2.connect(dsn, **kwargs)
    conn.autocommit = kwargs.pop('autocommit', True)
    return conn


def async_connect(dsn, **kwargs):
    '''Make asynchronous connection via `psycopg2.connect()`. Accepts all the arguments of
    the original function.

    Parameters
    ----------
    dsn : str
        DSN of a database to connect to

    Returns
    -------
    psycopg2.connection
        asynchronous connection established
    '''

    kwargs['async_'] = True
    conn = psycopg2.connect(dsn, **kwargs)
    wait(conn)
    return conn


def execute_and_wait(conn, query, vars=None, timeout=None):
    with closing(conn.cursor()) as cursor:
        cursor.execute(query, vars=vars)
        wait(conn, timeout)


def begin(conn, timeout=None):
    '''Begin the transaction. Should be used for the asynchronous connection since
    original `psycopg2.connection` methods do not work for connection in asynchronous mode.
    The call would block until db response received.

    Parameters
    ----------
    conn : psycopg2.connection
        connection to start transaction on
    timeout : double, optional
        timeout for IO (the default is None, which is infinity)

    '''

    execute_and_wait(conn, 'BEGIN;', timeout=timeout)


def commit(conn, timeout=None):
    '''Commit the transaction changes. Should be used for the asynchronous connection since
    original `psycopg2.connection` methods do not work for connection in asynchronous mode.
    The call would block until db response received.

    Parameters
    ----------
    conn : psycopg2.connection
        connection to commit transaction changes on
    timeout : double, optional
        timeout for IO (the default is None, which is infinity)

    '''

    execute_and_wait(conn, 'COMMIT;', timeout=timeout)


def rollback(conn, timeout=None):
    '''Rollback the transaction changes. Should be used for the asynchronous connection since
    original `psycopg2.connection` methods do not work for connection in asynchronous mode.
    The call would block until db response received.

    Parameters
    ----------
    conn : psycopg2.connection
        connection to rollback transaction changes on
    timeout : double, optional
        timeout for IO (the default is None, which is infinity)

    '''

    execute_and_wait(conn, 'ROLLBACK;', timeout=timeout)


def get_locks(conn):
    '''Request locks information about connection transaction

    Parameters
    ----------
    conn : psycopg2.connection
        Connection to examine.

    Returns
    -------
    [namedtuple('Record', ['locktype', 'objid', 'mode', 'transactionid', 'granted'])]
        Information about current locks of a transaction being performed on a given connection.
    '''

    query = 'SELECT locktype, objid, mode, transactionid, granted FROM pg_locks WHERE pid=%(pid)s AND NOT granted'
    with connect(conn.dsn) as info_conn:
        with closing(info_conn.cursor(cursor_factory=psycopg2.extras.NamedTupleCursor)) as cur:
            cur.execute(query, dict(pid=conn.get_backend_pid()))
            return cur.fetchall()


def is_locked(conn):
    '''Indicates if transaction being performed on a given connection
    is locked by another transaction

    Parameters
    ----------
    conn : psycopg2.connection
        connection to examine

    Returns
    -------
    bool
        `True` - if transaction on a given connection is locked by another transaction,
        `False` - otherwise.
    '''

    return int(len(get_locks(conn)) > 0)


default_schemas = ('code', 'impl')


def reflect_db(conn, schemas=None):
    '''Reflect database schemas into python objects hierarchy for a given connection.
    All the reflected entities will relate to and communicate through the connection.
    There is no reflection sharing between connections.

    Example
    -------
    There are three schemas in a database `code`, `impl`, `mops`, and
    there are composite types impl.composite, code.another_composite
    and two functions `impl.foo()`, `code.bar()` and two tables
    `mops.table_one`, `mops.table_two`.

    So after call

        db = reflect_db(connection)

    the hierarchy will be made and next rows would be valid

        db.code.bar()
        db.code.another_composite

        db.impl.foo()
        db.impl.composite

        db.mops.table_one
        db.mops.table_two

    Parameters
    ----------
    conn : psycopg2.connection
        connection which will be used within a database reflection
    schemas : list of strings, optional
        list of schemas to reflect (the default is None, which is `['code', 'impl']`)

    Returns
    -------
    object
        reflected database
    '''

    def select(query):
        with closing(conn.cursor(cursor_factory=psycopg2.extras.NamedTupleCursor)) as cur:
            try:
                cur.execute(query, vars={'schemas': schemas})
                if conn.async_:
                    wait(conn)
                return cur.fetchall()
            except:
                print(cur.query)
                raise

    if schemas is None:
        schemas = default_schemas

    decls_query = '''
        SELECT (SELECT nspname FROM pg_namespace WHERE oid = pronamespace) AS schema, proname AS name,
            (SELECT to_json(array_agg(json_build_object('name', arg_name, 'type', nspname || '.' || typname)
                                ORDER BY rn))
                -- proallargtypes - if all the arguments are IN arguments, this field will be null
                -- proargnames - Note that subscripts correspond to positions of proallargtypes not proargtypes.
                FROM unnest(coalesce(proallargtypes, proargtypes), proargnames, proargmodes) WITH ORDINALITY AS x(arg_oid, arg_name, arg_mode, rn)
                JOIN pg_type ON (pg_type.oid = arg_oid)
                JOIN pg_namespace ON (typnamespace = pg_namespace.oid)
                WHERE arg_mode IS NULL  -- If all the arguments are IN arguments, this field will be null
                OR arg_mode IN ('i', 'v')) AS args, proretset AS multi_row
        FROM pg_proc
        WHERE pronamespace IN (
            SELECT oid
            FROM pg_namespace
            WHERE nspname IN %(schemas)s);
    '''

    composites_query = '''
        SELECT (SELECT nspname FROM pg_namespace WHERE oid = typnamespace) AS schema, typname AS name,
            (SELECT COUNT(*)>0 FROM pg_tables AS t WHERE
                    typnamespace IN (SELECT oid FROM pg_namespace WHERE nspname = t.schemaname)
                    AND t.tablename = typname) AS is_table
        FROM pg_type
        WHERE typtype = 'c'
        AND typnamespace IN (SELECT oid FROM pg_namespace WHERE nspname IN %(schemas)s)
    '''

    decls = select(decls_query)

    composites = select(composites_query)

    out_schemas = {s: {} for s in schemas}

    def add_to_schema(item):
        out_schemas[item.schema][item.name] = item

    for d in decls:
        try:
            add_to_schema(SqlFunction(conn, d))
        except:
            print(d)
            raise

    for c in composites:
        try:
            add_to_schema(Table(conn, c) if c.is_table else Composite(conn, c))
        except:
            print(c)
            raise

    return namedtuple('Schemas', out_schemas.keys())(
        *[namedtuple('Schema_'+schema, item.keys())(
            *item.values()
        ) for schema, item in out_schemas.items()]
    )
