# coding: utf-8

from collections import namedtuple

PGLock = namedtuple(
    "PGLock",
    ("lock_type", "obj_id", "mode", "transaction_id", "granted",)
)

PGLock.TX_LOCK = 'transactionid'


def strip_q(q):
    return " ".join(l.strip() for l in q.split('\n'))


# 47.59. pg_locks
# http://www.postgresql.org/docs/9.3/static/view-pg-locks.html
# When one transaction finds it necessary
# to wait specifically for another transaction,
# it does so by attempting to acquire share lock on the other transaction ID
# (either virtual or permanent ID depending on the situation).
# That will succeed only when the other transaction
# terminates and releases its locks.
#
# Although tuples are a lockable type of object,
# information about row-level locks is stored on disk, not in memory,
# and therefore row-level locks normally do not appear in this view.
# If a transaction is waiting for a row-level lock,
# it will usually appear in the view as waiting for
# the permanent transaction ID of the current holder of that row lock.

class Diag(object):

    def __init__(self, conn):
        self.conn = conn

    @staticmethod
    def get_pid(pidded):
        if isinstance(pidded, int):
            return pidded
        if hasattr(pidded, 'pid'):
            return pidded.pid
        raise RuntimeError("Can't get get from {0}".format(pidded))

    def get_locks(self, pidded):
        cur = self.conn.cursor()
        cur.execute(
            strip_q("""SELECT locktype, objid, mode, transactionid, granted
                         FROM pg_locks
                        WHERE pid=%(pid)s"""),
            dict(pid=self.get_pid(pidded))
        )
        self.conn.wait()

        return [PGLock(*r) for r in cur.fetchall()]
