# coding: utf-8

from future.moves.itertools import zip_longest

from abc import ABCMeta
import psycopg2.extensions as PGE

from .common import MissedTypeArgument


class TypeBuildError(RuntimeError):
    pass


class BaseAdapted(object):
    __metaclass__ = ABCMeta
    __slots__ = ()

    def __conform__(self, proto):
        if proto is PGE.ISQLQuote:
            return self


class BaseAdaptedComposite(BaseAdapted):
    __metaclass__ = ABCMeta

    def __init__(self, **kwargs):
        self._conn = None
        for a in self.__slots__[1:]:
            try:
                setattr(self, a, kwargs[a])
            except KeyError:
                if hasattr(self, 'defaults'):
                    def_val = self.defaults[a]  # pylint: disable=E1101
                    if callable(def_val):
                        def_val = def_val()
                    setattr(self, a, def_val)
                else:
                    raise MissedTypeArgument(
                        "Can't find %s field for %s, kwargs=%r" % (
                            a, self.__class__.__name__, kwargs
                        )
                    )

    @classmethod
    def fields(cls):
        return cls.__slots__[1:]

    def __lt__(self, other):
        if type(self) != type(other):
            raise TypeError('{} can be compared only with objects of same type, got: {}'.format(self, other))
        for (lkey, lval), (rkey, rval) in zip_longest(self.as_dict().items(), other.as_dict().items()):
            assert lkey == rkey
            if lval < rval:
                return True
            if lval > rval:
                return False
        return False

    def __eq__(self, other):
        if not hasattr(other, 'as_dict'):
            raise TypeError('Expect object with as_dict, got %r' % other)
        return all(
            # Same key, same value
            left_item == right_item
            for left_item, right_item in zip_longest(self.as_dict().items(), other.as_dict().items())
        )

    def as_dict(self):
        return dict((a, getattr(self, a)) for a in self.fields())

    def prepare(self, conn):
        self._conn = conn

    def getquoted(self):
        # COPY-PASTE from psycopg2.extras.SQL_IN
        pobjs = [PGE.adapt(getattr(self, o)) for o in self.fields()]
        if self._conn is not None:
            for obj in pobjs:
                if hasattr(obj, 'prepare'):
                    obj.prepare(self._conn)
        qobjs = [o.getquoted() for o in pobjs]
        qform = [
            b'(',
            b', '.join(qobjs),
            b')::',
            self.pg_type_name.encode('utf-8')  # pylint: disable=E1101
        ]
        return b''.join(qform)

    def __repr__(self):
        return u"%s(%s)" % (
            self.__class__.__name__,
            u", ".join([
                u"%s=%r" % (k, getattr(self, k))
                for k in self.fields()
            ]))


class BaseAdaptedList(BaseAdapted):
    __slots__ = ('seq',)
    __metaclass__ = ABCMeta

    def __init__(self, seq):
        self.seq = list(seq)

    def __eq__(self, other):
        return self.seq == other.seq

    def getquoted(self):
        return PGE.adapt(self.seq).getquoted() + b'::' \
            + bytearray(self.pg_elem_type_name, encoding='utf-8') + b'[]'  # pylint: disable=E1101


# Example:
# StoreCoordinates = _make_composite(
#     'StoreCoordinates', [
#         'fid', 'tid',
#         'seen', 'deleted',
#         'st_id', 'received_date',
#         'size', 'attributes',
#         'pop_uidl'
#     ],
#     'code.store_coordinates'
# )
def _make_composite(name, fields, pg_type_name):
    return type(
        name,
        (BaseAdaptedComposite,),
        dict(
            __slots__=tuple(['_conn'] + fields),
            pg_type_name=pg_type_name
        )
    )


class SmallInt(BaseAdapted):
    def __init__(self, e):
        self.e = e

    def getquoted(self):
        return PGE.adapt(self.e).getquoted() + b'::smallint'


class ListOfSmallInt(BaseAdaptedList):
    __slots__ = ('seq',)
    pg_elem_type_name = 'smallint'


class ListOfMessageNumeric(BaseAdaptedList):
    __slots__ = ('seq',)
    pg_elem_type_name = 'numeric'
