# coding: utf-8

try:
    from itertools import zip_longest
except ImportError:
    # Python2
    from itertools import izip_longest as zip_longest

from .selected import BaseSelectedType


def _compare_base_selected(left, right, verbose):
    if not left.__slots__ == right.__slots__:
        return
    explanation = ['Comparing %r instances' % left.__class__]
    common_values = []
    for attr in left.__slots__:
        lv = getattr(left, attr)
        rv = getattr(right, attr)
        if lv != rv:
            explanation += [
                'Different %s.%s %r != %r' % (
                    left.__class__.__name__, attr, lv, rv)
            ]
        else:
            common_values.append(
                '%s=%r' % (attr, lv)
            )
    if verbose:
        explanation += [
            'Common attributes: (%s)' % ', '.join(common_values)
        ]
    return explanation


def _compare_list_of_base_selected(left, right, verbose):
    same = []
    difference = []
    for index, (lv, rv) in enumerate(zip_longest(left, right)):
        if lv != rv:
            if lv is None or rv is None:
                bk, bv = ('left', rv) if lv is None else ('right', lv)
                difference.append(
                    'Got `None` for %s at index %d, other: %r ' % (
                        bk, index, bv))
            else:
                difference.append('Difference at index %d' % index)
                difference += _compare_base_selected(lv, rv, verbose)
        elif verbose:
            same.append('Same at index %d' % index)
            same.append(repr(lv))
    return difference + same


def _is_base_select(obj):
    return isinstance(obj, BaseSelectedType)


def _is_list_of_base_select(obj):
    return isinstance(obj, list) \
        and all(_is_base_select(el) for el in obj)


def base_selected_repr_compare(left, right, verbose):
    '''helper for readable display difference in BaseSelectedType'''
    if _is_base_select(left) and _is_base_select(right):
        return _compare_base_selected(left, right, verbose)
    elif _is_list_of_base_select(left) and _is_list_of_base_select(right):
        return _compare_list_of_base_selected(left, right, verbose)


def repr_compare(left, right, verbose=False):
    repr_diff = base_selected_repr_compare(left, right, verbose)
    if repr_diff:
        return '\n'.join(repr_diff)
    return 'left: %r right %r' % (left, right)
