# coding: utf-8
from abc import ABCMeta

from future.moves.itertools import zip_longest

from .common import MissedTypeArgument


class BaseSelectedType(object):
    __metaclass__ = ABCMeta
    __slots__ = ()
    _defaults = {}

    def __init__(self, **kwargs):
        for k in self.__slots__:
            if k in kwargs:
                setattr(self, k, kwargs[k])
            elif k in self._defaults:
                def_val = self._defaults[k]
                if callable(def_val):
                    def_val = def_val()
                setattr(self, k, def_val)
            else:
                raise MissedTypeArgument(
                    "Can't find %s field for %s, kwargs=%r, defaults=%r" % (
                        k, self.__class__.__name__, kwargs, self._defaults
                    )
                )

    def __repr__(self):
        attrs = []
        for a in self.__slots__:
            attrs.append('%s=%s' % (a, repr(getattr(self, a))))

        return '%s(%s)' % (
            self.__class__.__name__,
            u', '.join(attrs)
        )

    @classmethod
    def argument_has_default_value(cls, argument):
        return argument in cls._defaults

    def as_dict(self):
        return dict([(k, getattr(self, k)) for k in self.__slots__])

    def __lt__(self, other):
        if type(self) != type(other):
            raise TypeError('{} can be compared only with objects of same type, got: {}'.format(self, other))
        for (lkey, lval), (rkey, rval) in zip_longest(self.as_dict().items(), other.as_dict().items()):
            assert lkey == rkey
            if lval < rval:
                return True
            if lval > rval:
                return False
        return False

    def __eq__(self, other):
        if not hasattr(other, 'as_dict'):
            raise TypeError('Expect object with as_dict, got %r' % other)
        try:
            return all(
                # Same key, same value
                left_item == right_item
                for left_item, right_item in zip_longest(self.as_dict().items(), other.as_dict().items())
            )
        except Exception:  # pylint: disable=broad-except
            return False

# Example:
# class Label(BaseSelectedType, LabelCompareMixin):
#     __slots__ = (
#         'lid', 'name', 'type', 'message_count',
#         'color', 'created', 'revision'
#     )
#
#     _defaults = {
#         'revision': 1,
#         'message_count': 0,
#         'color': None,
#         'created': datetime.now
#     }
