from werkzeug import routing
from werkzeug.exceptions import HTTPException, NotFound
from werkzeug.routing import Rule
from werkzeug.wrappers import Request, Response

from pymdb.tools import generate_st_id


class MulcagateApp(object):
    url_map = routing.Map([
        Rule('/', endpoint='describe'),
        Rule('/_list', endpoint='list'),

        Rule('/gate/put/<suid>', endpoint='put', methods=['POST']),
        Rule('/gate/get/<stid>', endpoint='get', methods=['GET']),
        Rule('/gate/del/<stid>', endpoint='del', methods=['GET']),
    ])

    def __init__(self):
        self.stids = {}

    def on_describe(self, request):
        return Response(str(self.url_map))

    def on_list(self, request):
        return Response(str(self.stids))

    def on_get(self, request, stid):
        if stid not in self.stids:
            return self.error_404(stid)
        return Response(self.stids[stid], status=200)

    def on_put(self, request, suid):
        stid = generate_st_id(suid)
        self.stids[stid] = request.get_data(as_text=True)
        return Response(stid)

    def on_del(self, request, stid):
        if stid not in self.stids:
            return self.error_404(stid)
        del self.stids[stid]
        return Response('ok')

    @staticmethod
    def error_404(stid):
        return Response('Stid not found: %s' % stid, status=404)

    def dispatch_request(self, request):
        adapter = self.url_map.bind_to_environ(request.environ)
        try:
            endpoint, values = adapter.match()
            return getattr(self, 'on_' + endpoint)(request, **values)
        except NotFound as e:
            return Response('Endpoint not found', status=500)
        except HTTPException as e:
            return e

    def __call__(self, environ, start_response):
        request = Request(environ)
        response = self.dispatch_request(request)
        return response(environ, start_response)
