from json import loads

from werkzeug import exceptions as http_exceptions
from werkzeug.utils import cached_property
from werkzeug.wrappers import Response, Request
from werkzeug.wsgi import responder

from fbb.blackbox_handler import BlackBoxHandler
from fbb.intranet_handler import IntranetHandler
from fbb.mailhost_handler import MailhostHandler
from fbb.sessionid_handler import SessionidHandler


class JSONRequest(Request):
    # accept up to 4MB of transmitted data.
    max_content_length = 1024 * 1024 * 4

    @cached_property
    def json(self):
        if self.headers.get('content-type') == 'application/json':
            return loads(self.data)
        return {}

    def get_arg(self, name, default=None):
        v = self.values.get(name, None)
        if v is not None:
            return v
        return self.json.get(name, default)

    def post_arg(self, name, default=None):
        return self.form[name] if name in self.form and self.form[name] is not None else default


class FakeBlackBoxApp(object):
    def __init__(self, user_db, sessionid_db):
        self.user_db = user_db
        self.sessionid_db = sessionid_db

    def index(self, request):
        '''
        http://{host}/blackbox - blackbox api
        http://{host}/mailhost - mailhosts api
        '''
        return Response(
            self.index.__doc__.format(host=request.host),
            mimetype='text/plain')

    def blackbox(self, request):
        return BlackBoxHandler(self.user_db, self.sessionid_db)(request)

    def mailhost(self, request):
        return MailhostHandler(self.user_db)(request)

    def intranet(self, request):
        return IntranetHandler(self.user_db)(request)

    def sessionid(self, request):
        return SessionidHandler(self.sessionid_db)(request)

    @responder
    def __call__(self, environ, start_response):
        request = JSONRequest(environ)
        handlers = {
            u'/': self.index,
            u'/blackbox': self.blackbox,
            u'/blackbox/': self.blackbox,
            u'/mailhost': self.mailhost,
            u'/mailhost/': self.mailhost,
            u'/1/bundle/account/register/intranet/': self.intranet,
            u'/save_sessionid': self.sessionid,
            u'/save_sessionid/': self.sessionid,
        }
        if request.path not in handlers:
            return http_exceptions.NotFound()
        return handlers[request.path](request)
