# -*- coding: utf-8 -*-
from werkzeug import exceptions as http_exceptions
from json import dumps

from fbb.blackbox_doc import BlackBoxDoc, ET


def error_response(error, text, use_json=False):
    if use_json:
        return dumps({
            "exception": {"value": text, "id": 2},
            "error": 'BlackBox error: %s' % error,
            "request_id": "f15976a4a08f27ce"
        })
    with BlackBoxDoc() as doc:
        doc.add('exception', id='2').text = text
        doc.add('error').text = 'BlackBox error: %s' % error
        return doc


def invalid_params_response(error, use_json=False):
    return error_response(error, 'INVALID_PARAMS', use_json)


def add_address_list(doc, emails):
    address_list = doc.add('address-list')
    address_attrs = {
        "default": "0",
        "validated": "1",
        "rpop": "0",
        "silent": "0",
        "unsafe": "0",
        "native": "1",
        "born-date": "2011-10-03 11:22:02",
    }
    default_address_attrs = dict(**address_attrs)
    default_address_attrs["default"] = "1"
    for i, email in enumerate(emails):
        # Mark first address as default, cause we don't care
        address = ET.Element('address', attrib=address_attrs if i else default_address_attrs)
        address.text = email
        address_list.append(address)


def add_address_list_json(info, emails):
    address_list = []
    for i, email in enumerate(emails):
        address = {
            "address": email,
            "validated": True,
            "default": False if i else True,
            "rpop": False,
            "silent": False,
            "unsafe": False,
            "native": True,
            "born-date": "2011-10-03 11:22:02",
        }
        address_list.append(address)
    info["address-list"] = address_list


def sessionid_response(xml):
    return error_response('xml is null', 'INVALID_DATA') if xml is None else xml


def userinfo_response_xml(user):
    if user is not None and user.userinfo_response is not None:
        return user.userinfo_response

    with BlackBoxDoc() as doc:
        uid_element = doc.add('uid', hosted='0')
        doc.add('karma', confirmed='0').text = '0'
        doc.add('karma_status').text = '0'
        if user:
            uid_element.text = str(user.uid)
            doc.add('have_password').text = '1'
            doc.add('login').text = user.login
            doc.add('have_hint').text = '0'
            sids = set(['2', '-'] + [str(s) for s in user.sids])
            for sid in sids:
                doc.add('dbfield', id='hosts.db_id.{}'.format(sid)).text = user.db
                doc.add('dbfield', id='subscription.suid.{}'.format(sid)).text = str(user.suid)
                doc.add('dbfield', id='subscription.login.{}'.format(sid)).text = user.login
                doc.add('dbfield', id='subscription.login_rule.{}'.format(sid)).text = "1"

            doc.add('dbfield', id='userinfo.lang.uid').text = "ru"
            doc.add('dbfield', id='account_info.country.uid').text = "ru"
            doc.add('dbfield', id="account_info.fio.uid").text = u"Резников Пётр"
            doc.add('dbfield', id="account_info.lang.uid").text = "ru"
            doc.add('dbfield', id="account_info.reg_date.uid").text = "2011-10-03 11:22:02"
            doc.add('dbfield', id="accounts.login.uid").text = user.login

            add_address_list(doc, user.emails)

            attributes = doc.add('attributes')
            if user.is_maillist:
                is_ml_attr = ET.Element('attribute', attrib={'type': '13'})
                is_ml_attr.text = '1'
                attributes.append(is_ml_attr)
                doc.add('dbfield', id='subscription.suid.1000').text = ''
            if user.orgid:
                orgid_attr = ET.Element('attribute', attrib={'type': '1031'})
                orgid_attr.text = str(user.orgid)
                attributes.append(orgid_attr)
        return doc


def userinfo_response_json(user):
    if user is not None and user.userinfo_response is not None:
        return user.userinfo_response

    info = {}
    info["id"] = user.uid if user else 0
    info["uid"] = {}
    info["karma"] = {"value": 0}
    info["karma_status"] = {"value": 0}
    if user:
        info["uid"] = {"hosted": 0, "value": user.uid}
        info["have_password"] = True
        info["login"] = user.login
        info["have_hint"] = False
        dbfields = {}
        sids = set(['2', '-'] + [str(s) for s in user.sids])
        for sid in sids:
            dbfields['hosts.db_id.{}'.format(sid)] = user.db
            dbfields['subscription.suid.{}'.format(sid)] = str(user.suid)
            dbfields['subscription.login.{}'.format(sid)] = user.login
            dbfields['subscription.login_rule.{}'.format(sid)] = "1"

        dbfields['userinfo.lang.uid'] = "ru"
        dbfields["userinfo.firstname.uid"] = u"Пётр"
        dbfields["userinfo.lastname.uid"] = u"Резников"
        dbfields['account_info.country.uid'] = "ru"
        dbfields["account_info.fio.uid"] = u"Резников Пётр"
        dbfields["account_info.lang.uid"] = "ru"
        dbfields["account_info.reg_date.uid"] = "2011-10-03 11:22:02"
        dbfields["accounts.login.uid"] = user.login
        info["dbfields"] = dbfields
        add_address_list_json(info, user.emails)
        info["attributes"] = {}
        if user.is_maillist:
            info["attributes"] = {
                "13": "1"
            }
            info["dbfields"]['subscription.suid.1000'] = ''
        if user.orgid:
            info["attributes"]["1031"] = str(user.orgid)
        info["aliases"] = {}
    return dumps({"users": [info]})


def userinfo_response(user, use_json):
    return userinfo_response_json(user) if use_json else userinfo_response_xml(user)


def unrecognized_request(request):  # pylint: disable=W0613
    return invalid_params_response(
        'Unrecognized request')


def not_implemented(request, description=None):  # pylint: disable=W0613
    return http_exceptions.NotImplemented(description)


def mailhost_error_response(error_text, error_id=2):
    with BlackBoxDoc() as doc:
        doc.add('status', id=error_id).text = error_text
        return doc


def unknown_op_handler(request):  # pylint: disable=W0613
    return mailhost_error_response('unknown op')


def ok_response():
    with BlackBoxDoc() as doc:
        doc.add('status', id=0).text = 'ok'
        return doc
