from werkzeug.wrappers import Response

from fbb.bb_response import invalid_params_response, userinfo_response, unrecognized_request, not_implemented, sessionid_response
from fbb.user import cast_by_value

import sys


class BlackBoxHandler(object):
    METHODS = [
        'userinfo',
        'login',
        'sessionid',
        'oauth',
        'lcookie',
        'litesession',
        'checkip',
        'sign',
        'checksignature'
    ]

    def __init__(self, user_db, sessionid_db):
        self.user_db = user_db
        self.sessionid_db = sessionid_db

    def userinfo(self, request):
        args = request.values
        response_json = 'format' in args is not None and args['format'] == 'json'
        if 'userip' not in args:
            return invalid_params_response(
                'Missing userip argument', response_json
            )
        user = None
        if 'login' in args and '@' in args['login']:
            user = self.user_db.get_by_email(args['login'])
            return userinfo_response(user, response_json)

        for by_key in ['uid', 'suid', 'login']:
            if by_key in args:
                by_value = args[by_key]
                try:
                    by_value = cast_by_value(by_key, by_value)
                except ValueError as exc:
                    return invalid_params_response(
                        'Malformed argument %r: %s' % (by_key, exc.message),
                        response_json
                    )
                user = self.user_db.get_by(by_key, by_value)
                break
        else:
            return invalid_params_response('Missing login argument', response_json)
        return userinfo_response(user, response_json)

    def sessionid(self, request):
        args = request.form

        if 'sessionid' not in args:
            return invalid_params_response(
                'Missing "sessionid" argument'
            )

        return sessionid_response(self.sessionid_db.get_sessionid(args['sessionid']))

    def __call__(self, request):
        method = request.values.get('method')
        handler = unrecognized_request

        if method in self.METHODS:
            try:
                handler = getattr(self, method)
            except AttributeError:
                return not_implemented(
                    request,
                    'method \'%s\' not implemented' % method)

        if sys.version_info.major == 2:
            response = str(handler(request))
        else:
            response = handler(request).__str__()
            if hasattr(response, 'decode'):
                response = response.decode('utf-8')

        return Response(response, mimetype='text/xml')
