import psycopg2 as pg

from fbb.user import User
from fbb.errors import NoSuchUserError, InvaidOldDB
from fbb.user_type import possible_uids, possible_suids, UserType


class UserDBFromPG(object):
    def __init__(self, dsn, is_corp):
        self.dsn = dsn
        self.is_corp = is_corp

    def get_by(self, key, value):
        with pg.connect(dsn=self.dsn) as conn:
            cur = conn.cursor()
            cur.execute(
                '''SELECT *, users.emails as emails
                     FROM fbb.users
                    WHERE {key} = %(value)s
                '''.format(key=key),
                dict(value=value)
            )
            user = cur.fetchone()
            if user:
                desc = [c.name.lower() for c in cur.description]
                return User(**dict(zip(desc, user)))

    def get_by_email(self, email):
        with pg.connect(dsn=self.dsn) as conn:
            cur = conn.cursor()
            cur.execute(
                '''SELECT *, users.emails as emails
                     FROM fbb.users
                    WHERE %(email)s = ANY(users.emails)
                ''',
                dict(email=email)
            )
            user = cur.fetchone()
            if user:
                desc = [c.name.lower() for c in cur.description]
                return User(**dict(zip(desc, user)))

    def set_db(self, suid, db, old_db):
        with pg.connect(dsn=self.dsn) as conn:
            cur = conn.cursor()
            cur.execute(
                '''SELECT db
                     FROM fbb.users
                    WHERE suid=%(suid)s
                      FOR UPDATE''',
                dict(suid=suid)
            )
            current_db = cur.fetchone()
            if current_db is None:
                raise NoSuchUserError(
                    'no such user suid: %r' % suid)
            current_db = current_db[0]
            if old_db is not None and current_db != old_db:
                raise InvaidOldDB(
                    'old_db_id=%r isn\'t equal '
                    'db_id_on_account=%r ' % (
                        old_db, current_db))
            cur.execute(
                '''UPDATE fbb.users
                      SET db=%(new_db)s
                    WHERE suid=%(suid)s''',
                dict(
                    suid=suid,
                    new_db=db,)
            )

    def next_user_ids(self, user_type):
        uids = []
        suids = []
        with pg.connect(dsn=self.dsn) as conn:
            cur = conn.cursor()
            cur.execute('SELECT uid, suid FROM fbb.users')
            for row in cur.fetchall():
                uids.append(row[0])
                suids.append(row[1])
        uid = next(u for u in possible_uids(user_type) if u not in uids)
        suid = next(su for su in possible_suids(user_type) if su not in suids)
        return uid, suid

    def create_user(self, login, db, user_type, userinfo_response, uid, orgid):
        user_id, suid = self.next_user_ids(user_type)
        if uid is None:
            uid = user_id

        with pg.connect(dsn=self.dsn) as conn:
            cur = conn.cursor()
            cur.execute(
                '''INSERT INTO fbb.users (uid, suid, login, db, is_corp, is_maillist, orgid, userinfo_response)
                   VALUES (%(uid)s, %(suid)s, %(login)s, %(db)s, %(is_corp)s, %(is_maillist)s, %(orgid)s, %(userinfo_response)s)''',
                dict(
                    uid=uid, suid=suid, login=login, db=db,
                    is_corp=self.is_corp, is_maillist=user_type == UserType.MAILLIST,
                    orgid=orgid, userinfo_response=userinfo_response
                ),
            )
        return uid, suid

    def check_user_exists(self, login):
        with pg.connect(dsn=self.dsn) as conn:
            cur = conn.cursor()
            cur.execute('SELECT count(*) FROM fbb.users WHERE login = %(login)s', dict(login=login))
            return cur.fetchone()[0] > 0
