# coding: utf-8

import logging
import os
import signal
import time
import threading
import Queue
# import cProfile
# from pstats import Stats

import gevent
import gevent.local
import gevent.pool
import gevent.socket
import simplejson

from apphost.api import service

log = logging.getLogger(__name__)
access_log = log.getChild('access')

log_local = gevent.local.local()
log_stats = None


def profiled(func):
    def wrap(*args, **kwargs):
        global log_stats, log_local

        now = time.time()

        # log_profiler = cProfile.Profile()
        # log_profiler.enable()
        retval = func(*args, **kwargs)
        # log_profiler.disable()

        # Trim to 3 three decimal places
        log_local.timings[func.__name__ + '_microsec'] = int((time.time() - now) * 1000 * 1000)
        # if log_stats:
        #     log_stats.add(log_profiler)
        # else:
        #     log_stats = Stats(log_profiler)
        return retval
    return wrap


class GeventWorker(threading.Thread):
    class StopItem(object):
        pass

    def __init__(self, handler, gevent_pool, max_queue_size):
        self.handler = handler
        self.q = Queue.Queue(max_queue_size)
        self.pool = gevent_pool
        self.rfd, self.wfd = os.pipe()
        super(GeventWorker, self).__init__()

    def start(self):
        return gevent.spawn(self._poll)

    def process(self, context, start_time):
        self._put(context, start_time)

    def stop(self):
        self._put(GeventWorker.StopItem, None)
        self.pool.join()

    def _put(self, context, start_time):
        try:
            self.q.put((context, start_time), timeout=0.01)
            os.write(self.wfd, 'A')
        except Queue.Full:
            self.handler.make_error(context, 'GeventWorker queue is full!')

    def _poll(self):
        while True:
            try:
                (context, start_time) = self.q.get_nowait()
            except Queue.Empty:
                gevent.socket.wait_read(self.rfd)
                os.read(self.rfd, 512)
                continue
            if context is GeventWorker.StopItem:
                return
            self.pool.spawn(self.handler, context, start_time)
            del context  # let context be garbage-collected


class AppHostHandler(object):
    u"""Subclass it and override _call() method."""
    def __init__(self, node_name, gevent_pool, max_queue_size=1000):
        self.node_name = node_name
        self.worker = GeventWorker(self.work, gevent_pool, max_queue_size)

    def __call__(self, context):
        start = time.time()
        self.worker.process(context, start)

    def start(self):
        self.worker.start()

    def stop(self):
        self.worker.stop()

    def work(self, context, start):
        log_local.timings = {}
        self._call(context)
        end = time.time()

        access_log.info(
            u'request_end',
            extra=self.log_extras(
                context,
                processing_time_microsec=int((end - start) * 1000 * 1000),
                **log_local.timings
            )
        )

    def _call(self, context):
        self.make_error(context, u'Not yet implemented: one must override AppHostHandler._call()')

    @staticmethod
    def log_extras(context, **kwargs):
        u"""
        Form "extra" logging attribute to pass into Logger.log() calls
         (see pymail.log_helpers.ExtraJsonFormatter for details)
        """
        return dict(
            apphost_request_id=str(context.request_id),
            **kwargs
        )

    def make_error(self, context, msg, **kwargs):
        context.add_item(self.node_name, simplejson.dumps(dict(error=msg, **kwargs), ensure_ascii=True))
        log.error('failure', extra=self.log_extras(context, error=msg, **kwargs))


class AppHostService(object):
    def __init__(self, port, handler, max_queue_size=100):
        """
        API to start/stop AppHost requests dispatching. Customize it with your handler.
        :type port: int
        :type handler: AppHostHandler
        """
        self.__loop = service.Loop(self.__on_shutdown)
        self.__loop.set_max_queue_size(max_queue_size)
        self.__handler = handler
        self.__loop.add(port, self.__handler)
        self.__running = False

    def start(self):
        if not self.__running:
            self.__handler.start()
            self.__loop.start(1)
            self.__running = True

    def stop(self):
        if self.__running:
            self.__loop.stop()
            self.__handler.stop()
            self.__running = False

    def __on_shutdown(self):
        # Функция будет вызвана при получении admin?action=shutdown, но до остановки.
        # Вызов произойтет в потоке библиотеки и если в ней вызвать Loop.stop(), то
        # произойдет deadlock.
        # Чтобы этого избежать генерируем сигнал прерывания
        os.kill(os.getpid(), signal.SIGINT)
