import logging

import gevent.pywsgi
from werkzeug import exceptions, routing, wrappers

from pymail.web_tools.exception_middleware import ExceptionMiddleware

log = logging.getLogger(__name__)
access_log = log.getChild('access')


class WebApplication(object):
    DEFAULT_URL_MAP = routing.Map([
        routing.Rule('/ping', endpoint='root'),
    ])

    def __init__(self, url_map=DEFAULT_URL_MAP):
        self.url_map = url_map

    @staticmethod
    def error_404():
        response = wrappers.Response('This page Not Found', mimetype='text/plain')
        response.status_code = 404
        return response

    def on_ping(self, request):
        return wrappers.Response('pong')

    def dispatch(self, request):
        adapter = self.url_map.bind_to_environ(request.environ)
        try:
            endpoint, values = adapter.match()
            return getattr(self, 'on_' + endpoint)(request, **values)
        except exceptions.NotFound:
            return self.error_404()
        except exceptions.HTTPException as e:
            return e

    def __call__(self, environ, start_response):
        request = wrappers.Request(environ)
        response = self.dispatch(request)
        return response(environ, start_response)


class HttpService(object):
    def __init__(self, port, web_app, gevent_pool):
        """
        Initializes HTTP server with given port and web.Application (fused with internal handlers).
        :type port: int
        :type web_app: WebApplication
        :type gevent_pool: gevent.pool.Pool
        """
        self.server = gevent.pywsgi.WSGIServer(
            listener=('', port),
            application=ExceptionMiddleware(web_app),
            spawn=gevent_pool,
            log=access_log,
            error_log=log,
        )

    def start(self):
        self.server.start()

    def stop(self):
        self.server.stop()
