import Queue
import logging
import threading
from datetime import datetime
from logging.handlers import WatchedFileHandler

import simplejson

from pymail import apphost_service, http_service

LOG_SQL = 15
LOG_SQL_NOTICE = 17
ADDITIONAL_LOG_LEVELS = {
    u'SQL': LOG_SQL,
    u'SQL_NOTICE': LOG_SQL_NOTICE,
}


def monkey_patch_logger():
    """Python 2.7 doesn't have logging.setLogRecordFactory method - we have to patch it by hand"""
    def makeRecord(self, name, level, fn, lno, msg, args, exc_info, func=None, extra=None):
        rv = self.__makeRecord(
            name, level, fn, lno, msg, args, exc_info, func, extra
        )
        if extra:
            rv.__dict__['extra'] = extra
        return rv

    logging.Logger.__makeRecord = logging.Logger.makeRecord
    logging.Logger.makeRecord = makeRecord


class ExtraJsonFormatter(logging.Formatter):
    """
    Format message as json entry, putting some additional predefined attributes.
    Also add anything that was provided as `extra` keyword parameter in `Logger.log()` call,
    if `monkey_patch_logger()` was called.
    """
    def format(self, record):
        message = super(ExtraJsonFormatter, self).format(record)
        entry = {
            'message': message,
            'datetime': datetime.fromtimestamp(record.created).isoformat(),
            'unixtime_ms': int(record.created * 1000),
            'logger_name': record.name,
        }
        if hasattr(record, 'extra'):
            entry.update(record.extra)
        return simplejson.dumps(entry)


class ProcessedWTH(WatchedFileHandler):
    FILE_HANDLERS = []

    class PoisonPill(object):
        pass

    def __init__(self, *args, **kwargs):
        super(ProcessedWTH, self).__init__(*args, **kwargs)
        self.filepath = args[0]
        self.q = Queue.Queue(1024*1024)
        self.thread = threading.Thread(target=self.qproc)
        self.thread.start()
        ProcessedWTH.FILE_HANDLERS.append(self)

    def join(self):
        self.q.put(ProcessedWTH.PoisonPill)
        self.thread.join()

    def qproc(self):
        while True:
            record = self.q.get()
            if record is ProcessedWTH.PoisonPill:
                break
            self._emit(record)

    def emit(self, record):
        try:
            self.q.put_nowait(record)
        except Queue.Full:
            print 'Queue for log [%s] is full' % self.filepath

    def _emit(self, record):
        return super(ProcessedWTH, self).emit(record)


def file_handler(path, level=None):
    hdlr = ProcessedWTH(path, mode='a', encoding='utf-8', delay=False)
    if level:
        hdlr.setLevel(level)
    hdlr.setFormatter(ExtraJsonFormatter())
    return hdlr


def init_logging(conf):
    for level_name, level in ADDITIONAL_LOG_LEVELS.items():
        logging.addLevelName(level, level_name)

    monkey_patch_logger()

    logging.getLogger().setLevel(
        logging.getLevelName(conf.get('log_level', 'info').upper())
    )

    http_access_logger = http_service.access_log
    http_access_logger.addHandler(file_handler(conf['http']['access_log']))
    http_access_logger.propagate = False

    apphost_access_logger = apphost_service.access_log
    apphost_access_logger.addHandler(file_handler(conf['apphost']['access_log']))
    apphost_access_logger.propagate = False

    error_handler = file_handler(conf['daemon']['error_log'], level=logging.ERROR)
    logging.getLogger().addHandler(error_handler)

    common_handler = file_handler(conf['daemon']['service_log'])
    logging.getLogger().addHandler(common_handler)


def join_logging(conf):
    map(ProcessedWTH.join, ProcessedWTH.FILE_HANDLERS)
    if apphost_service.log_stats:
        apphost_service.log_stats.dump_stats(conf.get('log_profiler_file', 'log_profiler.stats'))
