# coding: utf-8

# http://lucumr.pocoo.org/2007/5/21/getting-started-with-wsgi/

from sys import exc_info
from traceback import format_tb

from psycopg2 import IntegrityError

from pymail.web_tools.json_helpers import jdumps


class ExceptionMiddleware(object):
    """The middleware we use."""
    def __init__(self, app, return_tb=True):
        self.app = app
        self.return_tb = return_tb

    def __call__(self, environ, start_response):
        """Call the application can catch exceptions."""
        appiter = None
        # just call the application and send the output back
        # unchanged but catch exceptions
        try:
            appiter = self.app(environ, start_response)
            for item in appiter:
                yield item
        # if an exception occours we get the exception information
        # and prepare a traceback we can render
        except Exception as exception:
            e_type, e_value, tb = exc_info()
            traceback = ['Traceback (most recent call last):']
            traceback += format_tb(tb)
            traceback.append('%s: %s' % (e_type.__name__, e_value))
            # we might have not a stated response by now. try
            # to start one with the status code 500 or ignore an
            # raised exception if the application already started one.
            try:
                start_response('500 INTERNAL SERVER ERROR', [
                               ('Content-Type', 'application/json')])
            except:
                pass
            if self.return_tb:
                yield jdumps(make_error(exception=exception, traceback=traceback))
            else:
                yield jdumps(make_error(exception=exception))

        # wsgi applications might have a close function. If it exists
        # it *must* be called.
        if hasattr(appiter, 'close'):
            appiter.close()


def make_error(exception, traceback=None):
    error_dict = dict(
        code=ERROR_CODES.get(exception.__class__) or 1,
        message=str(exception),
    )
    if traceback:
        error_dict['traceback'] = traceback
    return dict(
        status='error',
        error=error_dict,
    )


ERROR_CODES = {
    Exception: 1,
    IntegrityError: 2,
}
