import json
from dataclasses import dataclass, field
from typing import Sequence, Mapping

import requests

from .detail import dc_from_fqdn


@dataclass
class Instance:
    ip: str
    fqdn: str
    state: str
    name: str
    id: int
    dc: str = field(init=False)

    def __post_init__(self):
        self.dc = dc_from_fqdn(self.fqdn)


@dataclass
class Component:
    name: str
    instances: Sequence[Instance]


Components = Mapping[str, Component]


@dataclass
class QloudInfoProvider:
    meta_file: str
    api_endpoint: str

    def local_dc(self) -> str:
        with open(self.meta_file, 'r') as f:
            local_meta = json.load(f)
            dc = local_meta['datacenter'].lower()
        return dc

    def local_ipv6(self) -> str:
        with open(self.meta_file, 'r') as f:
            local_meta = json.load(f)
            ipv6 = local_meta['ipv6'].lower().split('/')[0]
        return ipv6

    def local_component(self) -> str:
        with open(self.meta_file, 'r') as f:
            local_meta = json.load(f)
            instance_name = local_meta['qloud_instance'].lower()
        return instance_name.rsplit('-', 1)[0]

    def components_info(self) -> Components:
        res = requests.get(self.api_endpoint, {'format': 'json', 'state': '*', 'component': '*'})
        components_info = json.loads(res.content.decode('utf-8'))['components']
        return {
            component_name: Component(
                name=component_name,
                instances=[Instance(**info) for info in instances_info]
            )
            for component_name, instances_info in components_info.items()
        }
