import filecmp
import logging
import os
import shutil
import subprocess
import sys
from random import randint
from time import sleep
from typing import Sequence

log = logging.getLogger(__name__)


def update_configs(config_dir, component_names: Sequence[str], dry_run: bool):
    replace_configs(
        config_dir, component_names, dry_run,
        old_prefix='.include.bak', new_prefix='.include.new', result_prefix='.include'
    )


def restore_configs(config_dir, component_names: Sequence[str], dry_run: bool):
    replace_configs(
        config_dir, component_names, dry_run,
        old_prefix='.include.new', new_prefix='.include.bak', result_prefix='.include'
    )


def replace_configs(config_dir, component_names: Sequence[str], dry_run: bool, old_prefix, new_prefix, result_prefix):
    if configs_are_equal(config_dir, component_names, result_prefix, new_prefix):
        log.info('No changes in config files')
        return

    log.info(f'Found changes in config files {result_prefix}, backing up current config using prefix {old_prefix} and '
             f'replacing it using prefix {new_prefix}')
    copy_configs(config_dir, component_names, result_prefix, old_prefix)
    copy_configs(config_dir, component_names, new_prefix, result_prefix)
    if test_nginx_config():
        log.info('Nginx config test successful')
        if dry_run:
            log.info(f'Dry-run mode requested, returning previous config files using prefix {old_prefix}')
            copy_configs(config_dir, component_names, old_prefix, result_prefix)
        else:
            reload_nginx()
    else:
        log.error(f'Nginx config test failed, returning previous config files using prefix {old_prefix}')
        copy_configs(config_dir, component_names, old_prefix, result_prefix)


def test_nginx_config():
    cmd = 'nginx -t'
    p = subprocess.Popen(cmd.split(), stdout=sys.stdout, stderr=sys.stderr)
    retcode = p.wait()
    return retcode == 0


def reload_nginx():
    cmd = 'nginx -s reload'
    log.info('Some random sleep')
    sleep(randint(1, 10))
    p = subprocess.Popen(cmd.split(), stdout=sys.stdout, stderr=sys.stderr)
    retcode = p.wait()
    log.info(f'Nginx reloaded with status {retcode}')


def copy_configs(config_dir, component_names: Sequence[str], src_postfix, dst_postfix):
    for name in component_names:
        config_basename = os.path.join(config_dir, name)
        src_config = config_basename + src_postfix
        dst_config = config_basename + dst_postfix

        try:
            shutil.copyfile(src_config, dst_config)
            log.info('Copied ' + src_postfix + ' to ' + dst_postfix + ' for ' + str(name))
        except FileNotFoundError:
            log.warn(f'File {src_config} not found')


def configs_are_equal(config_dir, component_names: Sequence[str], src_postfix, dst_postfix):
    for name in component_names:
        config_basename = os.path.join(config_dir, name)
        src_config = config_basename + src_postfix
        dst_config = config_basename + dst_postfix
        if not os.path.exists(src_config):
            log.warning(f'src config {src_config} not found')
            return False

        eq = filecmp.cmp(src_config, dst_config, shallow=False)
        if eq is False:
            return False
    return True
