from __future__ import absolute_import, annotations

import json
from enum import Enum
from typing import Any, Callable, Dict, Optional, Type, Union, cast

import aiohttp_swagger
from aiohttp import hdrs, web
from apispec import APISpec, Path
from apispec.ext import marshmallow as apispec_marshmallow

from sendr_aiohttp.handler import MethodSchema


class SwaggerJSONEncoder(json.JSONEncoder):
    def default(self, o: Any) -> Any:
        if isinstance(o, Enum):
            return o.value
        return o


def swagger_dumps(data: Any) -> str:
    return json.dumps(data, cls=SwaggerJSONEncoder)


def safe_issubclass(cls, class_or_tuple):
    try:
        return issubclass(cls, class_or_tuple)
    except TypeError:
        return False


def get_view_methods(view_cls: Type[web.View]) -> Dict[str, Any]:
    http_methods = [
        meth for meth in map(str.lower, hdrs.METH_ALL)
        if hasattr(view_cls, meth)
    ]
    handlers = {
        meth: getattr(view_cls, meth)
        for meth in http_methods
    }
    return handlers


class ApplicationSpecUpdater:
    """
    Defines some common steps in generating API spec from application instance.
    """

    _LOCATION_MAP = {
        'match_info': 'path',
        'json': 'body',
    }

    app: web.Application
    spec: APISpec

    def update_spec(self) -> None:
        for route in self.app.router.routes():
            path = self.create_path(route)
            self.add_path(path)

    def add_path(self, path: Path) -> None:
        if path.path and path.operations:
            self.spec.add_path(path)

    def create_path(self, route: web.AbstractRoute) -> Path:
        """
        Given route create path spec.
        """

        if route.resource is None:
            return Path()

        return Path(
            path=route.resource.canonical,
            operations=self.get_path_operations(route),
        )

    def get_path_operations(self, route: web.AbstractRoute) -> dict:
        if not safe_issubclass(route.handler, web.View):
            return {}

        operations = {}

        handler = cast(Type[web.View], route.handler)

        for meth in get_view_methods(handler):
            operation_spec = self.get_operation_spec(handler, meth)
            if operation_spec:
                operations[meth] = operation_spec

        return operations

    def get_operation_spec(self, view_cls: Type[web.View], method: str) -> dict:
        operation: dict = {}

        meth_handler = getattr(view_cls, method)
        method_schema = MethodSchema.get_method_schema(meth_handler)

        if method_schema is None:
            return operation

        # inject parameters so MarshMallow Plugin will process them
        operation['parameters'] = [
            {
                'schema': schema,
                'in': self.translate_parameter_location(location),
            }
            for location, schema in method_schema.request_schema.items()
        ]

        if method_schema.response_schema is not None:
            schema_name = type(method_schema.response_schema).__name__
            self.spec.definition(schema_name, schema=method_schema.response_schema)
            responses = operation.setdefault('responses', {})
            responses[200] = {
                'schema': {'$ref': f'#/definitions/{schema_name}'}
            }

            operation['produces'] = ['application/json']

        if meth_handler.__doc__:
            doc_lines = meth_handler.__doc__.strip().splitlines()

            summary = doc_lines[0] if doc_lines else ''
            description = '\n'.join(doc_lines[1:]).strip()

            operation.update({
                'summary': summary,
                'description': description,
            })

        return operation

    def translate_parameter_location(self, location: str) -> str:
        return self._LOCATION_MAP.get(location, location)

    def __init__(self, app: web.Application, spec: APISpec):
        self.app = app
        self.spec = spec


TVM_SECURITY_DEFINITION = {
    'TVMTicket': {
        'type': 'apiKey',
        'in': 'header',
        'name': 'X-Ya-Service-Ticket',
    }
}


def add_security_definitions(spec: APISpec, security_definitions: dict) -> None:
    spec.options.setdefault('securityDefinitions', {}).update(security_definitions)

    security = spec.options.setdefault('security', [])

    if not security:
        security.append({})

    security[-1].update({key: [] for key in security_definitions})


def create_apispec(
    app: web.Application,
    title: str,
    version: str,
    security_definitions: Optional[dict] = None,
    get_spec_updater: Optional[Callable[[web.Application, APISpec], ApplicationSpecUpdater]] = None,
) -> APISpec:
    spec = APISpec(
        title=title,
        version=version,
        plugins=(
            apispec_marshmallow.MarshmallowPlugin(),
        ),
    )

    if security_definitions is None:
        security_definitions = TVM_SECURITY_DEFINITION

    add_security_definitions(spec, security_definitions)

    if get_spec_updater is None:
        app_updater = ApplicationSpecUpdater(app, spec)
    else:
        app_updater = get_spec_updater(app, spec)

    app_updater.update_spec()

    return spec


def spec_to_dict(spec: APISpec) -> dict:
    spec_dict = spec.to_dict()
    # Warmup lazy properties: this affects dumping via ujson module
    repr(spec_dict)
    return spec_dict


def setup_swagger_route(
    app: web.Application,
    spec: APISpec,
    swagger_route: str,
    swagger_route_name: Optional[str] = None,
) -> None:
    async def swagger_handler(_: web.Request) -> web.Response:
        return web.json_response(
            spec_to_dict(spec),
            headers={'Access-Control-Allow-Origin': '*'},
            dumps=swagger_dumps,
        )

    app.router.add_route(
        method='GET',
        path=swagger_route,
        handler=swagger_handler,
        name=swagger_route_name,
    )


def setup_swagger(
    app: web.Application,
    spec: Union[APISpec, dict],
    ui_version: Optional[int] = None,
    api_base_url: str = '/',
) -> None:
    if isinstance(spec, APISpec):
        spec_dict = spec_to_dict(spec)
    else:
        spec_dict = spec

    aiohttp_swagger.setup_swagger(
        app,
        ui_version=ui_version,
        swagger_info=spec_dict,
        api_base_url=api_base_url,
    )
