from aiohttp import web


class APIException(web.HTTPError):
    error_code = 'internal_error'
    default_message = 'Internal server error'

    def __init__(self, *, params=None, message=None):
        super().__init__()
        self.params = params or {}
        self.message = message or self.default_message
        self.status_code = self.status_code or 500


class MethodNotAllowed(web.HTTPBadRequest, APIException):
    error_code = 'method_not_allowed'
    default_message = 'Method {} is not allowed'

    def __init__(self, method, **kwargs):
        super().__init__(message=self.default_message.format(method), **kwargs)


class PermissionDenied(web.HTTPForbidden, APIException):
    error_code = 'permission_denied'
    default_message = 'Permission denied'


class InvalidRequest(web.HTTPBadRequest, APIException):
    error_code = 'invalid_params'
    default_message = 'Invalid request parameters'


class InvalidResponse(web.HTTPInternalServerError, APIException):
    error_code = 'invalid_response'
    default_message = 'Invalid response'


class ObjectNotFound(web.HTTPNotFound, APIException):
    error_code = 'not_found'
    default_message = 'Object not found'


class ObjectAlreadyExists(web.HTTPConflict, APIException):
    error_code = 'already_exists'
    default_message = 'Object already exists'


class ActionException(Exception):
    default_message = 'Action error'

    def __init__(self, *, orig_exc=None, message=None):
        self.orig_exc = orig_exc
        self.message = message or self.default_message
