from __future__ import annotations

import asyncio
from types import MethodType
from typing import Any, AsyncIterable, Callable, Dict, Optional, Type, TypeVar

import ujson
from aiohttp import web
from marshmallow import Schema
from webargs.aiohttpparser import AIOHTTPParser as BaseParser

from sendr_core import BaseAction
from sendr_core.exceptions import BaseCoreError

ViewMethType = TypeVar('ViewMethType', bound=MethodType)


class MethodSchema:
    SCHEMA_ATTRIBUTE = 'schema'

    SCHEMA_LOCATIONS = (
        'json',
        'match_info',
        'query',
        'form',
        'headers',
    )

    def __init__(self):
        self._request_schema: Dict[str, Schema] = {}
        self._response_schema: Optional[Schema] = None

    @classmethod
    def get_method_schema(cls, method: MethodType) -> Optional[MethodSchema]:
        return getattr(method, cls.SCHEMA_ATTRIBUTE, None)

    @classmethod
    def get_or_create_method_schema(cls, view_method: MethodType) -> MethodSchema:
        if not hasattr(view_method, cls.SCHEMA_ATTRIBUTE):
            setattr(view_method, cls.SCHEMA_ATTRIBUTE, MethodSchema())
        return getattr(view_method, cls.SCHEMA_ATTRIBUTE)

    @property
    def request_schema(self) -> Dict[str, Schema]:
        return self._request_schema.copy()

    def add_request_schema(self, schema: Schema, location: str) -> None:
        assert location in self.SCHEMA_LOCATIONS, f'Location "{location}" is invalid'
        assert location not in self._request_schema, f'Request schema for location "{location}" is already defined'
        self._request_schema[location] = schema

    @property
    def response_schema(self) -> Optional[Schema]:
        return self._response_schema

    @response_schema.setter
    def response_schema(self, schema: Schema) -> None:
        assert self._response_schema is None, 'Response schema is already defined'
        self._response_schema = schema


def request_schema(schema: Schema, location: str) -> Callable[[ViewMethType], ViewMethType]:
    def view_decorator(view_method: ViewMethType) -> ViewMethType:
        method_schema = MethodSchema.get_or_create_method_schema(view_method)
        method_schema.add_request_schema(schema, location)
        return view_method

    return view_decorator


def response_schema(schema: Schema) -> Callable[[ViewMethType], ViewMethType]:
    def view_decorator(view_method: ViewMethType) -> ViewMethType:
        method_schema = MethodSchema.get_or_create_method_schema(view_method)
        method_schema.response_schema = schema
        return view_method

    return view_decorator


class BaseHandler(web.View):
    PARSER = BaseParser()

    @property
    def app(self) -> web.Application:
        return self.request.app

    def _get_or_create_method_schema(self) -> MethodSchema:
        handler_cls = self.request.match_info.handler
        method = getattr(handler_cls, self.request.method.lower())
        return MethodSchema.get_or_create_method_schema(method)

    async def _log_failed_request_parse(self, schema: Schema, location: str, exception: Exception) -> None:
        pass

    async def parse_request(self, schema: Schema, location: str) -> dict:
        try:
            data = await self.PARSER.parse(schema, self.request, (location,))
        except Exception as exc:
            await self._log_failed_request_parse(
                schema=schema,
                location=location,
                exception=exc,
            )
            raise
        if not isinstance(data, dict):
            data = {type(data).__name__.lower(): data}
        return data

    async def get_data(self) -> dict:
        if asyncio.iscoroutinefunction(self._get_or_create_method_schema):
            method_schema = await self._get_or_create_method_schema()  # type: ignore
        else:
            method_schema = self._get_or_create_method_schema()

        data = {}
        for location, location_schema in method_schema.request_schema.items():
            location_data = await self.parse_request(location_schema, location)
            data.update(location_data)
        return data

    @staticmethod
    def make_schema_response(data: Any,
                             schema: Schema = None,
                             *,
                             content_type: str = 'application/json',
                             headers: Optional[dict] = None,
                             status: int = 200,
                             ) -> web.Response:
        if schema is not None:
            data, _ = schema.dump(data)

        return web.Response(
            text='' if status == 204 else ujson.dumps(data),
            content_type=content_type,
            headers=headers,
            status=status,
        )

    def make_response(self, data: Any, **kwargs: Any) -> web.Response:
        method_schema = self._get_or_create_method_schema()
        return self.make_schema_response(data, schema=method_schema.response_schema, **kwargs)

    async def make_stream_response(self,
                                   headers: dict,
                                   stream: AsyncIterable[bytes],
                                   ) -> web.StreamResponse:
        response = web.StreamResponse(headers=headers)
        await response.prepare(self.request)
        async for chunk in stream:
            await response.write(chunk)
        return response

    def _core_exception_result(self, exc: BaseCoreError) -> None:
        raise NotImplementedError

    async def run_action_setup(self, action_cls: Type[BaseAction], params: Optional[dict] = None) -> None:
        pass

    async def run_action(self, action_cls: Type[BaseAction], params: Optional[dict] = None) -> Any:
        params = params or {}
        await self.run_action_setup(action_cls, params)

        action = action_cls(**params)  # type: ignore

        try:
            return await action.run()
        except BaseCoreError as exc:
            self._core_exception_result(exc)
