from asyncio import CancelledError

from aiohttp import web
from aiohttp.log import server_logger

from .exceptions import APIException
from .response import error_response


def _convert_http_reason(reason):
    return '_'.join(reason.lower().split())


def _get_logger(request):
    return request.get('logger') or server_logger


async def middleware_response_formatter(app, handler):
    async def _handler(request):
        try:
            return await handler(request)
        except APIException as exc:
            return error_response(exc.status_code, exc.message, exc.error_code, params=exc.params)
        except web.HTTPError as exc:
            error_code = _convert_http_reason(exc._reason)
            message = exc._reason

            return error_response(exc.status_code, message, error_code)
        except CancelledError:
            _get_logger(request).info('Cancelled error')
            raise
        except Exception as exc:
            if not app.debug:
                _get_logger(request).exception('Unhandled exception')
                return error_response(500, 'Unhandled exception', 'unhandled_exception')
            else:
                raise exc
    return _handler
