from typing import Any, Optional

import ujson as json
from aiohttp import web

from .schemas import BaseSchema

__all__ = (
    'json_response',
    'schema_response',
    'error_response'
)


def json_response(response: Any,
                  status: str = 'ok',
                  status_code: int = 200,
                  headers: Optional[dict] = None,
                  content_type: str = 'application/json') -> web.Response:
    return web.Response(
        text=json.dumps({'status': status, 'response': response}),
        status=status_code,
        headers=headers,
        content_type=content_type
    )


def schema_response(response: Any,
                    schema: BaseSchema, *,
                    status: int = 200,
                    headers: Optional[dict] = None,
                    **kwargs: Any) -> web.Response:
    response = schema.dump_raising(response, **kwargs)
    return json_response(response, status='ok', status_code=status, headers=headers)


def error_response(status_code: int, message: str, error_code: str, params: Optional[dict] = None) -> web.Response:
    return json_response({
        'code': error_code,
        'message': message,
        'params': params or {}
    }, status='error', status_code=status_code)
