from typing import ClassVar, Iterable

from aiohttp.web import UrlDispatcher

from .exceptions import MethodNotAllowed

ALLOWED_METHODS = ["GET", "POST", "PUT", "DELETE", "*"]


class Url:
    def __init__(self, path, handler, name=None, method='*'):
        self.path = path
        self.handler = handler
        self.name = name
        self.method = method

    def validate(self):
        if self.method not in ALLOWED_METHODS:
            raise MethodNotAllowed(f"Method {self.method} is not allowed")


class PrefixedUrl(Url):
    PREFIX: ClassVar[str]

    def __init__(self, path, handler, name=None, method='*'):
        super().__init__(self.PREFIX + path, handler, name, method)


class BaseUrlDispatcher(UrlDispatcher):
    def add_routes(self, urls: Iterable[Url]) -> None:  # type: ignore
        assert isinstance(urls, (list, tuple))

        for url in urls:
            assert isinstance(url, Url)
            url.validate()
            self.add_route(url.method, url.path, url.handler, name=url.name)
