def get_alembic_command(db_module):
    """
        Создает команду для запуска alembic в каталоге
        с описанием миграций базы
        :param db_module: модуль с опсианием метаданных и alembic.ini внутри каталога
    """

    import click

    @click.command(context_settings=dict(
        allow_extra_args=True,
        ignore_unknown_options=True,
        help_option_names=()))
    @click.pass_context
    def alembic(ctx):
        """Run alembic db migration tool.

        """
        import os
        import sys

        import alembic.config

        # Changing current dir to db module dir
        os.chdir(db_module.__path__[0])

        # Run alembic with command arguments
        result = alembic.config.main(argv=ctx.args, prog='manage.py alembic')
        sys.exit(result)

    return alembic


def assert_db_consistent(db_module, alembic_cfg='alembic.ini'):
    """
        Проверяет, текущие миграции и их соответствие базе
        :param db_module: модуль с опсианием метаданных и alembic.ini внутри каталога

    """
    import os

    from alembic.config import Config
    from alembic.runtime.environment import EnvironmentContext
    from alembic.script import ScriptDirectory

    # Changing current dir to db module dir
    os.chdir(db_module.__path__[0])

    config = Config(file_=alembic_cfg)

    script = ScriptDirectory.from_config(config)

    def get_current():
        cur_rev = None

        def save_current(rev, ctx):
            nonlocal cur_rev
            cur_rev = script.get_all_current(rev)
            return ()

        with EnvironmentContext(
            config,
            script,
            fn=save_current
        ):
            script.run_env()

        return cur_rev

    heads = script.get_revisions('heads')

    assert len(heads) == 1, 'Got {} heads'.format(len(heads))

    current = get_current()
    assert len(current) == 1, 'Got {} db revisions'.format(len(current))

    head_revision = tuple(heads)[0].revision
    current_revision = tuple(current)[0].revision

    assert head_revision == current_revision, 'Got head:"{}" and current:"{}'.format(head_revision, current_revision)
