from dataclasses import MISSING, asdict, dataclass, fields, is_dataclass
from datetime import datetime
from enum import Enum
from typing import Any, Callable, Dict, Iterable, Optional

from sqlalchemy.sql import ColumnElement, FromClause, func

from .types import Entity, OptEntityType, OptStrList, ValuesMapping


@dataclass
class FieldMapper:
    field_name: str
    column_name: str
    converter: Callable


class SelectableDataMapper:
    """
    Класс предназначен для помощи в формировании запроса и маппинга
    полученных данных в dataclass

    Преобразование данных производится путем вызова экземпляра класса (см. __call__)
    Для того, чтобы маппинг работал корректно, необходимо в запросе использовать поля из
    свойства columns. Это надо сделать чтобы можно было использовать маппер со строками
    результата, полученного join'нами с другими таблицами

    Предпочтительный метод использования - создать унаследованный класс
    и переопределить в нем атрибуты entity_class и selectable. Как альтернатива можно передать
    `entity_class` и `selectable` в конструктор для AdHoc создания на базе подзапроса.

    В случае, если пара `entity_class` и `selectable` будет использоваться в запросе несколько раз,
    то необходимо воспользоваться параметром `label_prefix`, установив его для каждого использования
    в разные значения.

    Для реализации специфической обработки некоторых полей можно реализовать
    методы `map_<field_name>` с одним параметром - значение поля
    ```
        def map_int_field(self, value):
            return value or 0

        map_enum_fied = EnumClass
    ```
    """
    entity_class: OptEntityType = None  # dataclass в который будет преобразовываться запись selectable
    selectable: Optional[FromClause] = None  # таблица или запрос, из которого надо преобразовывать записи

    def __init__(self, *, entity_class: OptEntityType = None, selectable: Optional[FromClause] = None,
                 label_prefix: str = ''):
        entity_class = entity_class if entity_class is not None else self.entity_class
        selectable = selectable if selectable is not None else self.selectable
        if selectable is None:
            raise ValueError('selectable is not provided')

        if not isinstance(entity_class, type) or not is_dataclass(entity_class):
            raise TypeError('entity_class should be dataclass type')

        self._entity_class = entity_class
        if label_prefix:
            self._label_prefix = f'{label_prefix}__{entity_class.__name__}'
        else:
            self._label_prefix = f'{entity_class.__name__}'

        self._mapping: Dict[str, FieldMapper] = {}
        select_cols = []

        for field in fields(entity_class):
            if field.name not in selectable.c:
                if field.default is MISSING and field.default_factory == MISSING:  # type: ignore
                    raise TypeError(f'Field {entity_class.__name__}.{field.name} not in selectable and has no default')
                continue
            col_name = f'{self._label_prefix}__{field.name}'
            converter = getattr(self, f'map_{field.name}', lambda x: x)

            self._mapping[field.name] = FieldMapper(
                field_name=field.name,
                column_name=col_name,
                converter=converter,
            )
            select_cols.append(selectable.c[field.name].label(col_name))
        self._select_cols = tuple(select_cols)

    @property
    def columns(self) -> Iterable[ColumnElement]:
        """
        Список колонок с переопределнными именами для конкретного маппера
        """
        return self._select_cols

    def get_corresponding_column_name(self, field_name: str) -> str:
        """Имя колонки, из которой будет взято значение для поля с указанным именем"""
        return self._mapping[field_name].column_name

    def __call__(self, row: ValuesMapping) -> Entity:
        """
        Сделать маппинг строки запроса в dataclass
        """
        params = {
            name: mapper.converter(row[mapper.column_name])
            for name, mapper in self._mapping.items()
        }
        return self._entity_class(**params)


class TableDataDumper:
    """
    Класс предназначен для помощи в преобразовании атрибутов датакласса
    в значения для использования в запросах к БД

    Предпочтительный метод использования - создать унаследованный класс
    и переопределить в нем атрибуты entity_class и table. В качествe альтернативы
    можно передать `entity_class` и `table` в конструктор.

    Так же в конструкторе можно определить набор полей, которые будут учитываться
    с помощью параметров `skip_fields` или `keep_fields`.

    Для реализации специфической обработки некоторых полей можно реализовать
    методы `dump_<field_name>` с одним параметром - значение поля
    ```
        @staticmethod
        def dump_created_at(self, value):
            return sa.func.now()
    ```
    """
    entity_class: OptEntityType = None  # dataclass из которого будут получаться значения
    table: Optional[FromClause] = None  # таблица для определения полей в запросе

    def __init__(self, *, entity_class: OptEntityType = None, table: Optional[FromClause] = None,
                 skip_fields: OptStrList = None, keep_fields: OptStrList = None):
        entity_class = entity_class if entity_class is not None else self.entity_class
        table = table if table is not None else self.table
        if table is None:
            raise ValueError('table is not provided')
        if keep_fields and skip_fields:
            raise ValueError('skip_fields and keep_fields are mutually exclusive')
        skip_fields = set(skip_fields or ())
        keep_fields = set((x.name for x in table.c if not keep_fields or x.name in keep_fields))

        self._dump_fields = [
            (field.name, self._get_dump(field))
            for field in fields(entity_class) if field.name in keep_fields and field.name not in skip_fields
        ]

    def __call__(self, obj: Any, skip_fields: OptStrList = None, keep_fields: OptStrList = None) -> ValuesMapping:
        """
        Создать набор значений из переданного объектов
        `skip_fields` или `keep_fields` ограничивает какие поля преобразовывать
        """
        if keep_fields and skip_fields:
            raise ValueError('skip_fields and keep_fields are mutually exclusive')
        skip_fields = set(skip_fields or ())
        keep_fields = set(keep_fields or ())
        has_keep = bool(keep_fields)
        return {
            field: dumper(getattr(obj, field))
            for field, dumper in self._dump_fields
            if field not in skip_fields and (not has_keep or field in keep_fields)
        }

    def _get_dump(self, field):
        return getattr(self, f'dump_{field.name}', self._default_dump)

    @classmethod
    def _default_dump(cls, value):
        if isinstance(value, Enum):
            return value.value
        if is_dataclass(value):
            return asdict(value, dict_factory=cls._json_dict)
        return value

    @classmethod
    def _json_dict(cls, items):
        return {
            k: cls._json_value_dump(v)
            for k, v in items
        }

    @classmethod
    def _json_value_dump(cls, value):
        if isinstance(value, datetime):
            return value.isoformat()
        if isinstance(value, Enum):
            return value.value
        return value


class TimeDumper:
    """
    Класс, определяющий преобразования для типовых полей с датами
    """

    def dump_modified_at(self, value):
        return func.now()

    def dump_created_at(self, value):
        return func.now()
