# encoding: utf-8

"""Модуль кастомизирует aiopg.sa.engine для обработки потери соединения с БД"""
import asyncio
from typing import Any, Optional

import psycopg2
# WARN импортируем protected из aiopg
from aiopg.sa.engine import _EngineAcquireContextManager


class EngineMixin:
    def using(self, name: Optional[str] = None) -> Any:
        return self


class EngineClosingAcquireContextManager(_EngineAcquireContextManager):
    """
    Контекст менеджер, закрывающий соединение при проблемах с БД
    В aiopg 0.14 изменился интерфейс контекстного менеджера, добавлено для совместимости.
    """
    if not hasattr(_EngineAcquireContextManager, '_conn'):
        @property
        def _conn(self):
            return self._obj

        @_conn.setter
        def _conn(self, value):
            self._obj = value

    @asyncio.coroutine
    def __aexit__(self, exc_type, exc, tb):
        reasons_for_closing = (
            psycopg2.InternalError, psycopg2.OperationalError,
            psycopg2.InterfaceError, psycopg2.ProgrammingError,
            TimeoutError,
        )

        if not self._conn.connection.closed and exc in reasons_for_closing:
            try:
                self._conn.connection.close()
            except (psycopg2.DatabaseError, TimeoutError):
                pass

        self._conn._transaction = None
        self._savepoint_seq = 0

        yield from self._pool.release(self._conn)

        self._pool = None
        self._conn = None
