import asyncio
import logging
import uuid
from contextlib import asynccontextmanager
from enum import Enum, unique
from typing import Any, AsyncContextManager, Callable, Dict, Optional, Type, cast

from aiohttp.connector import TCPConnector
from aiopg.sa.connection import SAConnection
from aiopg.sa.engine import Engine

from sendr_aiopg.engine import EngineMixin
from sendr_interactions.base import AbstractInteractionClient
from sendr_qlog import LoggerContext

from .single import create_engine as create_single_engine  # noqa

logger = logging.getLogger(__name__)


# https://a.yandex-team.ru/arc/trunk/arcadia/toolbox/pg-pinger/README.md#hosts
@unique
class Preset(Enum):
    MASTER = 'master'
    SYNC = 'sync'
    SYNC_LOCAL = 'sync-local'
    LOCAL = 'local'
    ACTUAL_LOCAL = 'actual-local'


class PgPingerClient(AbstractInteractionClient[dict]):
    DEBUG = False
    REQUEST_RETRY_TIMEOUTS = (0.1, 0.2)
    SERVICE = 'pg_pinger'

    async def hosts(self, preset: Preset) -> dict:
        return await self.get(
            'hosts',
            f'{self.BASE_URL}/hosts',
            params={'preset': preset.value},
        )


WRAPPER_CTX_TYPE = Optional[Callable[
    [AsyncContextManager[SAConnection]], AsyncContextManager[SAConnection]
]]


class LazyEngine(EngineMixin):
    # возможность внедрить в LazyEngine вспомогательный контекст, например для снятия метрик
    acquire_connection_context_wrapper: WRAPPER_CTX_TYPE

    def __init__(self,
                 pg_pinger_cls: Type[PgPingerClient],
                 close_connector: bool = False,
                 *args: Any,
                 **kwargs: Any,
                 ):
        self._pg_pinger_cls = pg_pinger_cls
        self._close_connector = close_connector
        self._logger = LoggerContext(logger, {})
        self._engines: Dict[str, Engine] = {}
        self._lock = asyncio.Lock()
        self._args = args
        self._kwargs = kwargs
        self.acquire_connection_context_wrapper = None

    @property
    def logger(self) -> LoggerContext:
        return self._logger

    async def _get_host(self, preset: Preset) -> str:
        with self.logger:
            request_id = f'lazy_engine_{uuid.uuid4().hex}'
            pg_pinger = self._pg_pinger_cls(self.logger, request_id)
            try:
                response = await pg_pinger.hosts(preset)
                return response['hosts'][0]['hostname']
            finally:
                await pg_pinger.close()

    async def _create_engine(self, host: str):  # type: ignore
        async with self._lock:
            if host in self._engines:
                return
            engine = await create_single_engine(*self._args, host=host, **self._kwargs).__aenter__()
            self._engines[host] = engine

    async def _get_engine(self, preset: Preset) -> Engine:
        host = await self._get_host(preset)
        if host not in self._engines:
            await self._create_engine(host)
        return self._engines[host]

    @asynccontextmanager
    async def acquire(
        self,
        preset: Preset = Preset.MASTER,
    ) -> SAConnection:
        engine = await self._get_engine(preset)

        ctx = engine.acquire()

        if self.acquire_connection_context_wrapper is not None:
            ctx = self.acquire_connection_context_wrapper(ctx)

        async with ctx as conn:
            # preset is needed to determine later if a connection is reusable
            conn.conn_preset = preset
            yield conn

    def close(self) -> None:
        for engine in self._engines.values():
            engine.close()

    async def wait_closed(self) -> None:
        for engine in self._engines.values():
            await engine.wait_closed()
        if self._close_connector:
            await self._pg_pinger_cls.CONNECTOR.close()

    def using(self, name: Optional[str] = None) -> 'PresetLazyEngine':
        preset = Preset(name) if name else Preset.MASTER
        return PresetLazyEngine(engine=self, preset=preset)


class PresetLazyEngine:
    def __init__(self, engine: LazyEngine, preset: Preset):
        self._engine = engine
        self._preset = preset

    def acquire(self) -> AsyncContextManager[SAConnection]:
        return self._engine.acquire(self._preset)


def create_engine(pg_pinger_url: Optional[str] = None,
                  pg_pinger_cls: Optional[Type[PgPingerClient]] = None,
                  close_connector: bool = False,
                  *args: Any,
                  **kwargs: Any,
                  ) -> LazyEngine:
    assert pg_pinger_url or pg_pinger_cls, 'PgPingerClient configuration is required.'

    if pg_pinger_cls is None:
        class Client(PgPingerClient):
            BASE_URL = cast(str, pg_pinger_url).rstrip('/')
            CONNECTOR = TCPConnector()

        pg_pinger_cls = Client
        close_connector = True  # Since tcp connector is initialized here, engine is responsible for closing it.

    return LazyEngine(pg_pinger_cls=pg_pinger_cls, close_connector=close_connector, *args, **kwargs)  # type: ignore
