from dataclasses import dataclass, fields, is_dataclass
from typing import Any, Callable, ClassVar, Dict, Protocol, Type, TypeVar, Union

from sendr_aiopg.storage.exceptions import StorageNotFound
from sendr_utils import without_none
from sendr_utils.schemas.base import BaseSchema


class EntityMeta(type):
    def __new__(mcs, name, bases, dct):
        new_cls = super().__new__(mcs, name, bases, dct)
        new_cls.DoesNotExist = type(f'{name}.DoesNotExist', (StorageNotFound,), {})
        return new_cls


class EntityProtocol(Protocol):
    DoesNotExist: ClassVar[Type[Exception]]


@dataclass
class Entity(metaclass=EntityMeta):
    DoesNotExist: ClassVar[Type[Exception]]


@dataclass
class JSONBEntity:
    schema: ClassVar[Type[BaseSchema]]

    def __post_init__(self) -> None:
        self._rest: Dict[str, Any] = {}

    @classmethod
    def from_jsonb(cls, jsonb_field: Dict[str, Any]) -> 'JSONBEntity':
        constructor_params = {
            field.name: jsonb_field.get(field.name)
            for field in fields(cls)
        }
        params_parsed = cls.schema().load(constructor_params).data
        if is_dataclass(params_parsed):
            instance = params_parsed
        else:
            instance: 'JSONBEntity' = cls(**params_parsed)  # type: ignore
        instance._rest = {
            key: jsonb_field[key]
            for key in jsonb_field
            if key not in constructor_params
        }
        return instance

    def to_jsonb(self, skip_none: bool = False) -> Dict[str, Any]:
        data = self.schema().dump(self).data
        for key, value in self._rest.items():
            data[key] = value
        return without_none(data) if skip_none else data

    @classmethod
    def from_dataclass(cls, dt: Callable[..., Any]) -> 'JSONBEntity':
        """
        Часто бывает нужно, что существующий датакласс нужно сделать JSONBEntity.
        Для этого создаётся наследник `class Child(Parent, JSONBEntity)`.
        Этот метод позволяет конвертировать Parent в Child.
        """
        constructor_params = dict(dt.__dict__)
        if isinstance(dt, JSONBEntity):
            rest = dt._rest
            constructor_params.pop('_rest')
        instance = cls(**constructor_params)  # type: ignore
        if isinstance(dt, JSONBEntity):
            instance._rest = rest
        return instance


class NotFetchedType:
    """
    Взято из Я.Оплаты.
    Предназначен для обозначения связанных сущностей, которые не были извлечены.
    """
    __slots__ = ()

    def __deepcopy__(self, memodict):
        # костыль для dataclasses.asdict - не нужно копировать незменяемую константу
        # пусть работает is
        return self

    def __bool__(self):
        return False


NOT_FETCHED = NotFetchedType()

_LazyEntity = TypeVar('_LazyEntity')

LazyField = Union[_LazyEntity, NotFetchedType]
