from enum import Enum, unique
from typing import ClassVar, Type

from aiopg.sa.connection import SAConnection
from psycopg2.errors import LockNotAvailable

from sendr_qlog import LoggerContext


class LockAlreadyAcquired(Exception):
    pass


class Lock:
    def __init__(self, lock_id: int, connection: SAConnection, logger: LoggerContext):
        self._lock_id: int = lock_id
        self._connection: SAConnection = connection
        self._logger: LoggerContext = logger

    async def acquire(self) -> None:
        try:
            await self._connection.execute(f'SELECT pg_advisory_xact_lock({self._lock_id})')
        except LockNotAvailable:
            raise LockAlreadyAcquired

    async def __aenter__(self) -> None:
        return await self.acquire()


@unique
class BaseLockEnum(Enum):
    pass


class StorageLock:
    LOCK_ENUM: ClassVar[Type[BaseLockEnum]]

    def __init__(self, connection, logger):
        self._connection = connection
        self._logger = logger

    def __getattr__(self, item):
        try:
            lock_enum = getattr(self.LOCK_ENUM, item)
        except AttributeError:
            raise
        return Lock(
            lock_id=lock_enum.value,
            connection=self._connection,
            logger=self._logger,
        )
