from typing import List, Optional, Set

from sendr_interactions.clients.blackbox import AbstractBlackBoxClient, BlackBoxInvalidSessionError
from sendr_interactions.clients.blackbox.entities import BlackBoxResult, OauthResult, SessionIdResult
from sendr_qlog import LoggerContext

from .entities import AuthenticationMethod, User
from .exceptions import AuthenticationException


class BlackboxAuthenticator:
    def __init__(
        self,
        client: AbstractBlackBoxClient,
        scopes: Optional[List[str]],
        authorization_header: Optional[str],
        session_id: Optional[str],
        default_uid: Optional[str],
        user_ip: str,
        host: str,
        logger: LoggerContext,
        allowed_oauth_client_ids: Optional[Set[str]],
    ):
        super().__init__()
        self.client = client
        self.scopes = scopes
        self.authorization_header = authorization_header
        self.session_id = session_id
        self.default_uid = default_uid
        self.user_ip = user_ip
        self.host = host
        self.logger = logger
        self.allowed_oauth_client_ids = allowed_oauth_client_ids

    async def by_authorization_header(self) -> OauthResult:
        assert self.authorization_header is not None
        try:
            auth_realm, oauth_token = self.authorization_header.split(' ', maxsplit=1)
        except ValueError:
            raise AuthenticationException(message='AUTHORIZATION_HEADER_MALFORMED')
        if auth_realm.lower() != 'oauth':
            raise AuthenticationException(message='INCORRECT_AUTH_REALM')

        result = await self.client.get_oauth_token_info(
            oauth_token=oauth_token,
            user_ip=self.user_ip,
            scopes=self.scopes,
            get_user_ticket=True,
        )

        if self.allowed_oauth_client_ids is not None:
            client_id = result.client_id
            if not client_id or client_id not in self.allowed_oauth_client_ids:
                with self.logger:
                    self.logger.context_push(
                        request_client_id=client_id,
                        allowed_client_ids=self.allowed_oauth_client_ids,
                    )
                    self.logger.warning('Unauthorized access from disallowed client')
                raise AuthenticationException()

        return result

    async def by_session_id(self) -> SessionIdResult:
        assert self.session_id is not None
        uid: Optional[int]
        try:
            uid = int(self.default_uid) if self.default_uid else None
        except ValueError:
            uid = None

        return await self.client.get_session_info(
            host=self.host,
            session_id=self.session_id,
            user_ip=self.user_ip,
            default_uid=uid,
            get_user_ticket=True,
        )

    async def get_user(self) -> User:
        try:
            result: BlackBoxResult
            if self.authorization_header:
                result = await self.by_authorization_header()
                auth_source = AuthenticationMethod.OAUTH
                self.logger.context_push(oauth_client=result.client_id)
            elif self.session_id:
                result = await self.by_session_id()
                auth_source = AuthenticationMethod.SESSION
            else:
                raise AuthenticationException(message='MISSING_CREDENTIALS')
            self.logger.context_push(uid=result.uid, login_id=result.login_id, auth_source=auth_source)
            if result.uid is None:
                raise AuthenticationException(message='OAUTH_TOKEN_WITHOUT_USER_CREDENTIALS')
            return User(
                uid=result.uid,
                tvm_ticket=result.tvm_ticket,
                login_id=result.login_id,
                auth_method=auth_source,
                is_yandexoid=result.is_yandexoid,
            )
        except BlackBoxInvalidSessionError as exc:
            with self.logger:
                self.logger.context_push(params=exc.params)
                self.logger.warning('Unauthorized access to API')
            raise AuthenticationException()
