import hmac
from dataclasses import dataclass
from logging import Logger
from typing import List, Optional

from sendr_utils import utcnow

from .entities import User
from .exceptions import InvalidCSRFTokenException


@dataclass
class CsrfSettings:
    keys: List[str]
    token_lifetime: int = 86400
    header_name: str = 'X-CSRF-TOKEN'
    check_token_without_uid: bool = False


class CsrfChecker:
    """
    The reference: https://github.yandex-team.ru/maps/csrf/blob/master/src/index.ts#L83
    """
    def __init__(
        self,
        user: User,
        yandexuid: Optional[str],
        csrf_token: Optional[str],
        logger: Logger,
        settings: CsrfSettings,
    ):
        self.user = user
        self.yandexuid = yandexuid
        self.token = csrf_token
        self.logger = logger
        self.settings = settings

    @staticmethod
    def generate_token(
        key: str,
        timestamp: int,
        user: User,
        yandexuid: str,
    ) -> str:
        message = f'{user.uid}:{yandexuid}:{timestamp}'
        hmac_obj = hmac.new(key=key.encode('utf-8'), msg=message.encode('utf-8'), digestmod='sha1')
        digest = hmac_obj.digest().hex()
        return f'{digest}:{timestamp}'

    def _is_token_valid(self, key: str, timestamp: int) -> bool:
        assert key, 'Key is empty somehow'
        assert self.yandexuid is not None, 'Yandexuid should not be None'

        valid_tokens = [
            self.generate_token(
                key=key,
                timestamp=timestamp,
                user=self.user,
                yandexuid=self.yandexuid,
            )
        ]

        if self.settings.check_token_without_uid:
            valid_tokens.append(
                self.generate_token(
                    key=key,
                    timestamp=timestamp,
                    user=User(0),
                    yandexuid=self.yandexuid,
                )
            )

        return self.token in valid_tokens

    def check(self) -> None:
        if not self.token:
            self.logger.warning('token is not present')
            raise InvalidCSRFTokenException(message='MISSING_TOKEN')
        try:
            timestamp = int(self.token.split(':')[1])
        except (ValueError, IndexError):
            self.logger.warning("Can't parse timestamp")
            raise InvalidCSRFTokenException(message='INVALID_TIMESTAMP')
        if timestamp < int(utcnow().timestamp()) - self.settings.token_lifetime:
            # timestamp токена может быть выписан по другим часам, нежели часы на нашем бэкенде.
            # Но если разница не более, чем LIFETIME, то должно быть нормально
            self.logger.warning('Token expired')
            raise InvalidCSRFTokenException(message='EXPIRED_TOKEN')
        if not self.yandexuid:
            self.logger.warning('yandexuid is not set')
            raise InvalidCSRFTokenException(message='MISSING_YANDEXUID')

        for key in self.settings.keys:
            if self._is_token_valid(key, timestamp):
                return

        self.logger.warning('Invalid token')
        raise InvalidCSRFTokenException()
