import logging
from typing import Any, Awaitable, Callable, Coroutine, List, Optional, Set, Type, TypeVar
from uuid import uuid4

from aiohttp import hdrs, web, web_urldispatcher

from sendr_interactions import RetryBudgetProtocol
from sendr_interactions.clients.blackbox import AbstractBlackBoxClient
from sendr_qlog.logging.adapters.logger import LoggerContext

from .blackbox import BlackboxAuthenticator
from .csrf import CsrfChecker, CsrfSettings
from .entities import User
from .exceptions import AuthenticationException

default_logger = logging.getLogger()

F = TypeVar("F", bound=Callable[..., object])
HandlerType = Callable[[web.Request], Awaitable[web.Response]]
MiddlewareType = Callable[[web.Request, HandlerType], Coroutine[Any, Any, web.StreamResponse]]


def skip_csrf_check(handler: F) -> F:
    handler.skip_csrf_check = True  # type: ignore
    return handler


def skip_authentication(handler: F) -> F:
    handler.skip_authentication = True  # type: ignore
    return handler


def optional_authentication(handler: F) -> F:
    handler.optional_authentication = True  # type: ignore
    return handler


def _has_attr(obj: Any, name: str) -> bool:
    """
    Middlewares are chained using `functools.partial`.
    Lifting from inner to outer callable is done only for `__dict__` contents.
    But `__dict__` does not contain base class attributes
    and in order to access them we traverse the `__wrapped__` attribute.
    """
    while obj is not None:
        if hasattr(obj, name):
            return True
        obj = getattr(obj, '__wrapped__', None)
    return False


def create_csrf_middleware(settings: CsrfSettings) -> MiddlewareType:
    safe_methods = {hdrs.METH_GET, hdrs.METH_HEAD, hdrs.METH_OPTIONS}

    @web.middleware
    async def csrf_middleware(request: web.Request, handler: HandlerType) -> web.StreamResponse:
        if isinstance(request.match_info, web_urldispatcher.MatchInfoError):
            # either 404 or 405
            return await handler(request)

        if request.method in safe_methods:
            return await handler(request)

        user: Optional[User] = request.get('user')
        if user is None or len(request.cookies) == 0:
            # Неаутентифицированные запросы и запросы без кук не нуждаются в csrf
            return await handler(request)

        if not _has_attr(handler, 'skip_csrf_check'):
            CsrfChecker(
                user=user,
                yandexuid=request.cookies.get('yandexuid'),
                csrf_token=request.headers.get(settings.header_name),
                logger=request.get('logger') or LoggerContext(default_logger, {}),
                settings=settings,
            ).check()

        return await handler(request)

    return csrf_middleware


def create_blackbox_middleware(
    client_cls: Type[AbstractBlackBoxClient],
    oauth_scopes: Optional[List[str]],
    host: str,
    retry_budget: Optional[RetryBudgetProtocol] = None,
    ignored_paths: Optional[Set[str]] = None,
    ignored_path_prefixes: Optional[Set[str]] = None,
    allowed_oauth_client_ids: Optional[Set[str]] = None,
) -> MiddlewareType:
    @web.middleware
    async def blackbox_middleware(request: web.Request, handler: HandlerType) -> web.StreamResponse:
        if isinstance(request.match_info, web_urldispatcher.MatchInfoError):
            # either 404 or 405
            return await handler(request)

        if (
            request.method == hdrs.METH_OPTIONS
            or ignored_paths and request.path in ignored_paths
            or ignored_path_prefixes and any(request.path.startswith(prefix) for prefix in ignored_path_prefixes)
            or _has_attr(handler, 'skip_authentication')
        ):
            return await handler(request)

        logger = request.get('logger') or LoggerContext(default_logger, {})
        request_id = request.get('request-id') or request.headers.get('X-Request-Id') or uuid4().hex
        user_ip = request.headers.get('X-Forwarded-For-Y') or request.headers.get('X-REAL-IP', request.remote)
        bb_client = client_cls(logger=logger, request_id=request_id, retry_budget=retry_budget)

        try:
            request['user'] = await BlackboxAuthenticator(
                client=bb_client,
                scopes=oauth_scopes,
                authorization_header=request.headers.get('Authorization'),
                session_id=request.cookies.get('Session_id'),
                default_uid=request.query.get('default_uid'),
                user_ip=user_ip,
                host=host,
                logger=logger,
                allowed_oauth_client_ids=allowed_oauth_client_ids,
            ).get_user()
        except AuthenticationException:
            if not _has_attr(handler, 'optional_authentication'):
                raise
        finally:
            await bb_client.close()

        return await handler(request)

    return blackbox_middleware
