import asyncio

import click


def shell_command(init_state, config=None):
    ensure_event_loop_policy_is_singletone()

    @click.command()
    def shell():
        """Start interactive shell."""
        params = dict(
            argv=[],
        )
        user_ns = {}
        if callable(init_state):
            user_ns = init_state()
        elif isinstance(init_state, dict):
            user_ns = init_state
        params['user_ns'] = user_ns
        if callable(config):
            params['config'] = config()

        from IPython import start_ipython
        start_ipython(**params)
    return shell


def admin_command():
    @click.command()
    def admin():
        """Sleep command for forever runnable admin container

        Usage: manage.py admin
        """
        print("Admin dummy command run")
        from time import sleep

        while True:
            sleep(10)
    return admin


def ensure_event_loop_policy_is_singletone():
    if not isinstance(asyncio.get_event_loop_policy(), SingletoneEventLoopPolicy):
        asyncio.set_event_loop_policy(SingletoneEventLoopPolicy())


class SingletoneEventLoopPolicy(asyncio.DefaultEventLoopPolicy):  # type: ignore
    """
    ipython своими грязными руками вызывает new_event_loop/set_event_loop направо и налево.
    Поэтому сделаем так, чтобы при вызове new_event_loop всегда возвращался один и тот же loop.
    Нам никогда не нужно два event loop'а. Даже в проде. Тем более, в shell. Так что, звучит безопасно.

    Немного подробнее про проблему:

    IPython умеет обрабатывать async/await благодаря фиче под названием "autoawait"
    https://ipython.readthedocs.io/en/stable/interactive/autoawait.html

    Работает так:
    1. Определяем, является ли код асинхронным. Это делается с помощью `compile`
    2. Запустим этот код некоторым образом. Например asyncio.get_event_loop().run_until_complete(code)

    То, как именно запускается код, инкапсулировано в config.InteractiveShell.loop_runner: Callable[Coroutine]
    В стандартном IPython REPL нет никакого background event loop, поэтому реализация по-умолчанию работала так:
    берем asyncio.get_event_loop() и исполняем его.

    Халява закончилась в https://a.yandex-team.ru/review/2242403/files/1#file-0-120143094:R26
    Теперь стандартный AsyncIORunner создаёт loop самостоятельно.
    Но мы сначала инициализируем наш фреймворк, а уже потом проваливаемся в ipython. Получается, при инициализации
    фреймворка используется один loop, а в ipython - уже другой loop. Asyncio отрицательно реагирует на такое
    (и правильно делает), срабатывают предохранители.
    """
    def new_event_loop(self):
        if self._local._loop is not None:
            return self._local._loop
        return super().new_event_loop()
