import asyncio
from copy import copy
from typing import ClassVar, Generic, NoReturn, TypeVar

from sendr_core.context import BaseCoreContext
from sendr_core.exceptions import BaseCoreError, CoreFailError
from sendr_qlog import LoggerContext

ContextType = TypeVar('ContextType', bound=BaseCoreContext)


class BaseAction(Generic[ContextType]):
    no_cancel: ClassVar[bool] = False
    context: ClassVar[ContextType]

    def __new__(cls, *args, **kwargs):
        init_args = copy(args)
        init_kwargs = copy(kwargs)
        instance = super().__new__(cls)
        instance._init_args = init_args
        instance._init_kwargs = init_kwargs
        return instance

    @property
    def logger(self) -> LoggerContext:
        return self.context.logger

    @property
    def request_id(self) -> str:
        return self.context.request_id

    async def _secure_run(self, coro):
        try:
            return await coro
        except BaseCoreError:
            raise
        except asyncio.CancelledError:
            raise
        except Exception as exc:
            self.logger.exception('Core failed')
            self._exception_result(exc)

    def _exception_result(self, exc: Exception) -> NoReturn:
        raise CoreFailError from exc

    async def _run(self):
        await self._secure_run(self.pre_handle())
        result = await self._secure_run(self.handle())
        await self._secure_run(self.post_handle())
        return result

    async def pre_handle(self) -> None:
        pass

    async def post_handle(self) -> None:
        pass

    async def handle(self):
        raise NotImplementedError

    async def run(self):
        loop = asyncio.get_event_loop()
        task = loop.create_task(self._run())
        if self.no_cancel:
            return await asyncio.shield(task)
        return await task
