import asyncio
from dataclasses import dataclass
from random import randint

from sendr_core import BaseAction as AbstractAction
from sendr_core import BaseCoreContext


# Giving action context instance to work with.
class BaseAction(AbstractAction):
    context = BaseCoreContext()


@dataclass
class AddAction(BaseAction):
    """Simple action, returns sum of numbers it's been given"""

    a: int
    b: int

    async def handle(self) -> int:
        # Action may access context request_id and even modify it, without
        # messing up context of actions executed concurrently.
        print(self.request_id, f'Add action called with {self.a}, {self.b}.')
        await asyncio.sleep(0)

        result = self.a + self.b
        await asyncio.sleep(0)

        print(self.request_id, f'{self.a} + {self.b} = {result}.')
        return result


async def handler(request_id, **kwargs):
    # Handler / middleware sets up context after it's been copied.
    BaseAction.context.request_id = request_id
    return await AddAction(**kwargs).run()


async def main():
    """
    Runs AddAction concurrently with different params and request_id.

    Output should look like (first number is request_id):
    ```
        0 Add action called with 9, 3.
        1 Add action called with 9, 8.
        2 Add action called with 2, 3.
        0 9 + 3 = 12.
        1 9 + 8 = 17.
        2 2 + 3 = 5.
    ```

    Try commenting out block that creates ContextVars, making context a global
    value (context.py). Output will be something like:
    ```
        0 Add action called with 8, 3.
        1 Add action called with 3, 3.
        2 Add action called with 4, 7.
        2 8 + 3 = 11.
        2 3 + 3 = 6.
        2 4 + 7 = 11.
    ```
    """
    loop = asyncio.get_event_loop()

    # Creating task copies context, so the original context is not modified.
    # aiohttp does that on it's own before calling handler.
    tasks = []
    for n in range(3):
        a = randint(1, 10)
        b = randint(1, 10)
        tasks.append(loop.create_task(handler(n, a=a, b=b)))

    await asyncio.wait(tasks)


if __name__ == '__main__':
    loop = asyncio.get_event_loop()
    loop.run_until_complete(main())
