from datetime import datetime
from io import BytesIO
from typing import Any, List, Mapping, Optional, Protocol, Tuple, Type, TypedDict, cast
from urllib.parse import unquote as urlunquote

import aiobotocore.client
import aiobotocore.paginate
import aiobotocore.response
import aiobotocore.session
from botocore import exceptions as boto_exceptions

from .exceptions import FileExists, FileNotFound, IsADirectory

DEFAULT_READ_SIZE = 5 * 1024 * 1024  # 5MB


class StorageClient:

    def __init__(self, bucket: str, s3client_creator: aiobotocore.session.ClientCreatorContext):
        if not bucket:
            raise ValueError('Bad bucket name')
        self._bucket = bucket
        self._client_creator = s3client_creator
        self._client_instance: Optional[aiobotocore.client.AioBaseClient] = None
        self._entered_once = False

    @property
    def _client(self) -> aiobotocore.client.AioBaseClient:
        assert self._client_instance is not None
        return self._client_instance

    # Прокси над методами botocore клиента
    async def __aenter__(self) -> 'StorageClient':
        assert not self._entered_once
        self._client_instance = await self._client_creator.__aenter__()
        self._entered_once = True
        return self

    async def __aexit__(self, exc_type: Type[Exception], exc_val: Exception, exc_tb: Any) -> None:
        await self._client.__aexit__(exc_type, exc_val, exc_tb)
        self._client_instance = None

    def close(self) -> None:
        return self._client.close()

    async def create_bucket(self) -> None:
        """
        Создание бакета
        """
        try:
            await self._client.create_bucket(Bucket=self._bucket)
        except boto_exceptions.ClientError as exc:
            if exc.response.get('Error', {}).get('Code') != 'NoBucketAlreadyOwnedByYou':
                raise

    async def drop_bucket(self) -> None:
        """
        Удаление бакета
        """
        uploads = await self.list_uploads()
        for upl in uploads:
            self.drop_upload(upl)
        try:
            await self._client.delete_bucket(Bucket=self._bucket)
        except boto_exceptions.ClientError as exc:
            if exc.response.get('Error', {}).get('Code') != 'NoSuchBucket':
                raise

    async def list_dir(self, path: str) -> Tuple[List[str], List[str]]:
        """
        Содержимое каталога
        :param path: путь к каталогу

        :return: список дочерних каталогов и список файлов
        """
        path = self._fix_dir_path(path)
        base_parts = path.split("/")[:-1]
        base_parts_len = len(base_parts)

        files = []
        dirs = set()
        async for page in self._list_objects_paginator.paginate(Bucket=self._bucket, Prefix=path):
            for obj in page.get('Contents', ()):
                obj_key = obj.get('Key')
                if not obj_key:
                    continue
                key_parts = urlunquote(obj_key).split('/')[base_parts_len:]
                if len(key_parts) == 1:
                    # File
                    files.append(key_parts[0])
                elif len(key_parts) > 1:
                    # Dir
                    dirs.add(key_parts[0])
        return sorted(dirs), files

    async def list_uploads(self) -> List['UploadView']:
        uploads = []
        async for page in self._uploads_paginator.paginate(Bucket=self._bucket):
            for upl in page.get('Uploads', ()):
                uploads.append(UploadView(upl))
        return uploads

    async def make_dir(self, path: str) -> None:
        """
        Создает каталог
        :param path: путь к создаваемому каталогу
        :raises FileExists: если уже существует файл с таким именем
        """
        file_path = self._fix_file_path(path)
        if await self._file_exists(file_path):
            raise FileExists(filename=file_path)
        path = self._fix_dir_path(path)
        if await self._dir_exists(path):
            return
        await self._client.put_object(Bucket=self._bucket, Key=path + '.fld')

    async def drop_dir(self, path: str) -> List[str]:
        """
        Удаляет каталог и все дочерние файлы и каталоги
        :param path: путь к каталогу
        :return: список удаленных объектов
        """
        path = self._fix_dir_path(path)

        objects = []
        async for page in self._list_objects_paginator.paginate(Bucket=self._bucket, Prefix=path):
            for obj in page.get('Contents', ()):
                obj_key = obj.get('Key')
                if obj_key:
                    objects.append(obj_key)

        result = await self._client.delete_objects(
            Bucket=self._bucket,
            Delete={
                'Objects': [
                    {'Key': key}
                    for key in objects
                ]
            }
        )
        return result.get('Deleted')

    async def drop_upload(self, upload_view: 'UploadView') -> None:
        await self._client.abort_multipart_upload(
            Bucket=self._bucket,
            Key=upload_view.key,
            UploadId=upload_view.upload_id,
        )

    async def drop_file(self, path: str) -> None:
        """
        Удаляет файл, если он существует
        :param path: путь к файлу
        :raises FileNotFound: если путь не указывает на файл
        """
        self._raise_not_file(path)
        if not path:
            raise FileNotFound()
        await self._client.delete_object(Bucket=self._bucket, Key=path)

    async def exists(self, path: str) -> bool:
        """
        Проверяет существование файла или папки
        :param path: путь к каталогу (если заканчивается на "/") или к файлу
        :return: True/False в зависимости от наличия файла/каталога
        """
        if not path or path.endswith('/'):
            return await self._dir_exists(path)
        return await self._file_exists(path)

    async def download(self, path: str) -> 'Reader':
        """
        Скачать файл
        :param path: путь к файлу
        :return: объект Reader с метаинформацией и содержимым файла для потокового чтения
        """
        self._raise_not_file(path)
        try:
            obj = await self._client.get_object(Bucket=self._bucket, Key=path)
        except boto_exceptions.ClientError as exc:
            if exc.response.get('Error', {}).get('Code') == 'NoSuchKey':
                raise FileNotFound(filename=path)
            raise
        return Reader(obj)

    def upload_stream(self, path: str, read_size: int = DEFAULT_READ_SIZE) -> 'Uploader':
        self._raise_not_file(path)
        return Uploader(self._bucket, self._client, path=path, read_size=read_size)

    @property
    def _list_objects_paginator(self) -> aiobotocore.paginate.AioPaginator:
        return self._client.get_paginator('list_objects')

    @property
    def _uploads_paginator(self) -> aiobotocore.paginate.AioPaginator:
        return self._client.get_paginator('list_multipart_uploads')

    def _fix_dir_path(self, path: str) -> str:
        if path and not path.endswith('/'):
            return path + '/'
        return path

    def _fix_file_path(self, path: str) -> str:
        while path.endswith('/'):
            path = path[:-1]
        return path

    def _raise_not_file(self, path: str) -> None:
        if path.endswith('/'):
            raise IsADirectory(filename=path)

    async def _file_exists(self, path: str) -> bool:
        try:
            await self._client.head_object(Bucket=self._bucket, Key=path)
            return True
        except boto_exceptions.ClientError as exc:
            if exc.response.get('Error', {}).get('Code') == '404':
                return False
            raise

    async def _dir_exists(self, path: str) -> bool:
        resp = await self._client.list_objects(Bucket=self._bucket, Prefix=path, MaxKeys=1)
        if resp.get('Contents'):
            return True
        return False


class UploaderChunk(TypedDict):
    ETag: str
    PartNumber: int


class ReadableBuffer(Protocol):
    def read(self, n: int = -1) -> bytes:
        raise NotImplementedError


class Uploader:
    """
    Класс-обертка надо операцией multipart upload.
    Позволяет писать в S3 частями и получать на выходе один файл

    Должен создаватья только клиентом S3 и использоваться в async with
    """

    # Минимальный размер не последней части в S3 - 5МБ. Здесь делаем чуть
    # больше.
    _min_buffer = 10 * 1024 * 1024

    def __init__(self, bucket: str, s3client: aiobotocore.client.AioBaseClient, *, path: str, read_size: int):
        self._bucket = bucket
        self._client = s3client
        self._read_size = read_size
        self._path = path

        self._upl_id: Optional[str] = None
        self._chunks: Optional[List[UploaderChunk]] = None
        self._buffer: Optional[BytesIO] = None

    async def __aenter__(self) -> 'Uploader':
        self._upl_id = await self._start_multipart()
        self._chunks = []
        self._buffer = BytesIO()
        return self

    async def __aexit__(self, exc_type: Type[Exception], exc_val: Exception, exc_tb: Any) -> None:
        if self._upl_id:
            assert self._buffer is not None
            if exc_type or (not self._chunks and not len(self._buffer.getvalue())):
                await self._abort()
            else:
                await self._complete()

    async def write(self, data: bytes) -> None:
        """
        Метод записывает бинарные данные
        :param data: бинарные данные для записи, ожидается объект bytes, bytearray или подобный
        """
        await self.write_stream(BytesIO(data))

    async def write_stream(self, buffer: ReadableBuffer) -> None:
        """
        Метод читает данные из переданного буфера и записывает частями.
        Буффер вычитывается до конца
        :param buffer: буффер с бинарными данными (BytesIO, файл...)
        """
        if not self._upl_id:
            raise ValueError('Upload not started')
        assert self._buffer is not None

        while True:
            read_maxsize = min(
                self._min_buffer - len(self._buffer.getvalue()),
                self._read_size,
            )
            chunk = buffer.read(read_maxsize)
            if not chunk:
                return
            self._buffer.write(chunk)
            await self._flush()

    async def _flush(self, force: bool = False) -> None:
        assert self._buffer is not None
        assert self._chunks is not None

        buf_len = len(self._buffer.getvalue())
        flush_needed = buf_len >= self._min_buffer or force
        if not flush_needed or not buf_len:
            return
        part_no = len(self._chunks) + 1
        try:
            resp = await self._client.upload_part(
                Bucket=self._bucket,
                Key=self._path,
                UploadId=self._upl_id,
                PartNumber=part_no,
                Body=self._buffer.getvalue()
            )
            self._chunks.append({'ETag': resp['ETag'], 'PartNumber': part_no})
            self._buffer = BytesIO()
        except boto_exceptions.ClientError as exc:
            if exc.response.get('Error', {}).get('Code') == 'NoSuchUpload':
                raise FileNotFound(filename=self._path)
            raise

    async def _start_multipart(self) -> str:
        resp = await self._client.create_multipart_upload(Bucket=self._bucket, Key=self._path)
        return resp['UploadId']

    async def _abort(self) -> None:
        try:
            await self._client.abort_multipart_upload(Bucket=self._bucket, Key=self._path, UploadId=self._upl_id)
        except boto_exceptions.ClientError:
            pass
        await self._close()

    async def _complete(self) -> None:
        try:
            await self._flush(force=True)
            await self._client.complete_multipart_upload(
                Bucket=self._bucket,
                Key=self._path,
                UploadId=self._upl_id,
                MultipartUpload={'Parts': self._chunks},
            )
        except boto_exceptions.ClientError as exc:
            if exc.response.get('Error', {}).get('Code') == 'NoSuchUpload':
                raise FileNotFound(filename=self._path)
            raise
        finally:
            await self._close()

    async def _close(self) -> None:
        self._upl_id = self._chunks = self._buffer = None

    def __str__(self) -> str:
        return f'S3 upload to {self._path} with id: {self._upl_id}'

    def __repr__(self) -> str:
        return str(self)


class Reader:
    """
    Обертка над загружаемым файлом. Добавляет метаинформацию к потоку файла
    Для корректной работы надо либо вычитывать, либо закрывать(могут быть warning'и) content.
    """

    def __init__(self, s3response: Mapping[str, Any]):
        self._content: aiobotocore.response.StreamingBody = cast(
            aiobotocore.response.StreamingBody,
            s3response.get('Body')
        )
        self._etag: str = cast(str, s3response.get('ETag'))
        self._size: int = cast(int, s3response.get('ContentLength'))
        self._mtime: datetime = cast(datetime, s3response.get('LastModified'))

    @property
    def etag(self) -> str:
        return self._etag

    @property
    def size(self) -> int:
        return self._size

    @property
    def mtime(self) -> datetime:
        return self._mtime

    @property
    def content(self) -> aiobotocore.response.StreamingBody:
        return self._content

    def __str__(self) -> str:
        return f'S3 file with etag {self._etag} of size {self._size} modified at {self._mtime}'

    def __repr__(self) -> str:
        return str(self)


class UploadView:

    def __init__(self, s3response: Mapping[str, Any]):
        self._key: str = cast(str, s3response.get('Key'))
        self._upload_id: str = cast(str, s3response.get('UploadId'))
        self._initated: datetime = cast(datetime, s3response.get('Initiated'))

    @property
    def key(self) -> str:
        return self._key

    @property
    def upload_id(self) -> str:
        return self._upload_id

    @property
    def initated(self) -> datetime:
        return self._initated

    def __str__(self):
        initiated = self._initated and self._initated.isoformat()
        return f'Multipart upload to {self._key} with {self._upload_id} started at {initiated}'

    def __repr__(self):
        return str(self)
