from typing import Optional

import aiobotocore.session
import botocore

from .client import StorageClient


class S3Storage:

    def __init__(
        self,
        host: str,
        *,
        bucket: str,
        access_key_id: str,
        secret_access_key: str,
        user_agent: str = 'Sendr',
        connect_timeout: int = 1,
        read_timeout: int = 10,
        maxsize: int = 50,
        retries: int = 2,
    ):
        self._host = host
        self._bucket = bucket

        self._access_key_id = access_key_id
        self._secret_access_key = secret_access_key

        self.config = botocore.client.Config(
            user_agent=user_agent,
            connect_timeout=connect_timeout,
            read_timeout=read_timeout,
            max_pool_connections=maxsize,
            retries=dict(max_attempts=retries)
        )

        self._session: Optional[aiobotocore.session.AioSession] = None

    @property
    def host(self) -> str:
        return self._host

    @property
    def bucket(self) -> str:
        return self._bucket

    @property
    def session(self) -> aiobotocore.session.AioSession:
        if self._session is None:
            self._session = aiobotocore.session.get_session()
        return self._session

    def acquire(self) -> StorageClient:
        boto_client = self.session.create_client(
            's3',
            endpoint_url=self._host,
            aws_access_key_id=self._access_key_id,
            aws_secret_access_key=self._secret_access_key,
            config=self.config
        )

        return StorageClient(self._bucket, boto_client)
