import json
from abc import ABCMeta
from typing import ClassVar

from aiohttp import ClientResponse, ContentTypeError, MultipartWriter

from sendr_interactions.base import AbstractInteractionClient

from .entities import UploadResult
from .exceptions import BaseAvatarsError


class AbstractAvatarsClient(AbstractInteractionClient, metaclass=ABCMeta):
    SERVICE: ClassVar[str] = 'avatars'

    NAMESPACE: ClassVar[str]
    READ_BASE_URL: ClassVar[str]
    WRITE_BASE_URL: ClassVar[str]

    async def _handle_response_error(self, response: ClientResponse) -> None:
        error_code = response.status
        exc_cls = BaseAvatarsError.get_exception_by_code(error_code)
        try:
            data = await response.json()
            message = data.get('description', 'No description present')
        except (ContentTypeError, json.JSONDecodeError):
            message = 'Unable to decode error json'
        await self._try_log_error_response_body(response)
        raise exc_cls(method=response.method, message=message)

    def read_url(self, relative_url: str) -> str:
        return self.endpoint_url(relative_url, base_url_override=self.READ_BASE_URL)

    def write_url(self, relative_url: str) -> str:
        return self.endpoint_url(relative_url, base_url_override=self.WRITE_BASE_URL)

    async def upload(self, data: bytes, image_name: str = '') -> UploadResult:
        """
        Для использования upload без указания image_name, в namespace должен быть включен auto-id.
        """

        with MultipartWriter('form-data') as mpwriter:
            part = mpwriter.append(data)
            part.set_content_disposition('form-data', filename='image', name='file')

        url = self.write_url(f'put-{self.NAMESPACE}/{image_name}')
        response = await self.post(
            'upload',
            url,
            data=mpwriter,
        )
        return UploadResult(
            namespace=self.NAMESPACE,
            group_id=str(response['group-id']),
            image_name=response['imagename'],
        )
