from __future__ import annotations

from typing import Any, ClassVar, Dict, Optional, Type

from sendr_interactions import exceptions as interaction_errors


class BaseAvatarsError(interaction_errors.BaseInteractionError):
    """
    Для ручки put вот такие коды:
    https://wiki.yandex-team.ru/mds/avatars/#kodyvozvrata7
    """
    _exceptions_by_code: ClassVar[Dict[int, Type[BaseAvatarsError]]] = {}

    def __init__(
        self, *,
        method: str,
        message: Optional[str] = None,
        params: Optional[Dict[str, Any]] = None,
    ) -> None:
        if params is not None:
            self.params = params
        super().__init__(service='avatars', method=method, message=message)

    def __init_subclass__(cls) -> None:
        code: int = getattr(cls, 'HTTP_STATUS_CODE', None)
        if code is not None:
            assert code not in BaseAvatarsError._exceptions_by_code
            BaseAvatarsError._exceptions_by_code[code] = cls

    @classmethod
    def get_exception_by_code(cls, http_status_code: int) -> Type[BaseAvatarsError]:
        return cls._exceptions_by_code.get(http_status_code, UnknownAvatarsError)


class UnknownAvatarsError(BaseAvatarsError):
    pass


class UnsupportedMediaTypeError(BaseAvatarsError):
    HTTP_STATUS_CODE = 415


class ForbiddenError(BaseAvatarsError):
    HTTP_STATUS_CODE = 403


class RateLimitError(BaseAvatarsError):
    HTTP_STATUS_CODE = 429
