from datetime import date
from typing import Optional

from sendr_interactions.clients.balance.base import BaseBalanceClient
from sendr_interactions.clients.balance.entities import Client
from sendr_utils import without_none


class BalanceClientClient(BaseBalanceClient):
    @staticmethod
    def _init_client(data: dict) -> Client:
        return Client(
            client_id=str(data['CLIENT_ID']),
            name=data['NAME'],
            email=data['EMAIL'],
            phone=data['PHONE'],
        )

    async def create_client(self, uid: int, client: Client) -> str:
        """
        Creates client which is not linked to uid by default.
        If client has client_id, updates existing instead.
        Returns it's id.
        """
        dict_data = {
            'NAME': client.name,
            'EMAIL': client.email,
            'PHONE': client.phone,
        }
        if client.client_id:
            dict_data['CLIENT_ID'] = client.client_id
        result = await self.request(
            'create_client',
            method_name='CreateClient',
            data=(str(uid), dict_data),
        )
        code, _, client_id = result
        self._handle_error_codes(code)
        return str(client_id)

    async def create_user_client_association(self, uid, client_id):
        """
        Links client to uid. UID can only be linked to one client.
        """
        result = await self.request(
            'create_user_client_association',
            method_name='CreateUserClientAssociation',
            data=(str(uid), client_id, str(uid)),
        )
        code, _ = result
        self._handle_error_codes(code)

    async def remove_user_client_association(self, uid, client_id):
        """
        Removes client-uid link.
        """
        result = await self.request(
            'remove_user_client_association',
            method_name='RemoveUserClientAssociation',
            data=(str(uid), client_id, str(uid)),
        )
        code, _ = result
        self._handle_error_codes(code)

    async def find_client(self, uid: int, total_timeout: Optional[float] = None) -> Optional[Client]:
        """
        API returns list of all clients linked to given uid. List is either empty or contains a single dict.
        Returns client if it exists, None otherwise.
        """
        result = await self.request(
            'find_client',
            method_name='FindClient',
            data=({'PassportID': str(uid)},),
            **self._get_timeout_kwargs(total_timeout)
        )
        code, _, clients_data = result
        self._handle_error_codes(code)
        return self._init_client(clients_data[0]) if clients_data else None

    async def link_integration_to_client(
        self, uid: int, client_id: str, integration_cc: str, configuration_cc: str, start_date: Optional[date] = None
    ) -> None:
        integration_data = without_none({
            'ClientID': client_id,
            'IntegrationCC': integration_cc,
            'ConfigurationCC': configuration_cc,
            'StartDate': start_date.isoformat() if start_date is not None else None,
        })
        await self.request(
            'link_integration_to_client',
            method_name='LinkIntegrationToClient',
            data=(str(uid), integration_data),
        )
