from dataclasses import dataclass
from datetime import datetime
from enum import Enum, unique
from typing import Optional


@unique
class PaymentType(Enum):
    POSTPAYMENT = 3


@unique
class OfferConfirmationType(Enum):
    NO = 'no'


@unique
class CollateralType(Enum):
    """Тип дополнительного соглашения"""
    GENERAL_TERMINATION = 90


@dataclass
class Client:
    """
    Балансовая сущность клиента. На самом деле, все поля опциональные.
    У созданного клиента обязателен только client_id.
    """
    name: str
    email: str
    phone: str
    client_id: str = ''


@dataclass
class Person:
    client_id: str

    fname: str  #
    lname: str
    email: str
    phone: str

    name: str
    longname: str
    inn: str
    ogrn: str

    person_id: Optional[str] = None
    kpp: Optional[str] = None  # required for companies, null for entrepreneurs

    mname: Optional[str] = None

    account: Optional[str] = None
    bik: Optional[str] = None

    # Должно выполняться has_legal_address
    legal_address_city: Optional[str] = None
    legal_address_home: Optional[str] = None
    legal_address_postcode: Optional[str] = None
    legal_address_street: Optional[str] = None
    legal_address: Optional[str] = None

    address_city: Optional[str] = None
    address_home: Optional[str] = None
    address_postcode: Optional[str] = None
    address_street: Optional[str] = None
    post_address: Optional[str] = None

    postcode: Optional[str] = None

    date: Optional[datetime] = None

    @property
    def has_explained_legal_address(self) -> bool:
        return all(
            (
                self.legal_address_postcode is not None,
                self.legal_address_city is not None,
                self.legal_address_street is not None,
                self.legal_address_home is not None,
            )
        )

    @property
    def has_legal_address(self) -> bool:
        return self.legal_address is not None or self.has_explained_legal_address

    @property
    def has_explained_post_address(self) -> bool:
        return not any((
            self.address_city is None,
            self.address_home is None,
            self.address_postcode is None,
            self.address_street is None,
        ))

    @property
    def has_post_address(self) -> bool:
        return self.post_address is not None or self.has_explained_post_address

    @property
    def has_postcode(self) -> bool:
        return self.get_postcode() is not None

    def get_postcode(self) -> Optional[str]:
        return self.postcode or self.address_postcode or self.legal_address_postcode
