from typing import ClassVar, Iterable, Optional, Tuple, Union

from sendr_interactions import exceptions as interaction_errors


class BaseBalanceError(interaction_errors.BaseInteractionError):
    _subclasses_by_code: dict = {}
    MESSAGE: str = 'Unknown balance error'
    CODES: ClassVar[Tuple[Union[int, str, None], ...]]

    def __init__(self, *, message=None):
        super().__init__(service='balance', method='POST', message=message or self.MESSAGE)

    def __init_subclass__(cls, **kwargs):
        for code in getattr(cls, 'CODES', ()):
            assert code not in cls._subclasses_by_code
            cls._subclasses_by_code[code] = cls

    @staticmethod
    def get_error_by_codes(codes: Iterable[Union[str, int, None]]) -> Optional['BaseBalanceError']:
        for code in codes:
            if code in BaseBalanceError._subclasses_by_code:
                return BaseBalanceError._subclasses_by_code[code]
        return None


class BalancePersonNotFound(BaseBalanceError):
    """Person not found in balance, e. g. when filter client contracts by non existing person_id."""
    MESSAGE = 'BALANCE_PERSON_NOT_FOUND'
    CODES = ('PERSON_NOT_FOUND',)


class BalanceClientNotFoundError(BaseBalanceError):
    MESSAGE = 'BALANCE_CLIENT_NOT_FOUND'
    CODES = ('CLIENT_NOT_FOUND',)


class BalanceUserAlreadyLinkedToClient(BaseBalanceError):
    MESSAGE = 'BALANCE_USER_ALREADY_LINKED_TO_CLIENT'
    CODES = (4008,)


class BalanceUserNotLinkedToClient(BaseBalanceError):
    MESSAGE = 'BALANCE_USER_NOT_LINKED_TO_CLIENT'
    CODES = (4009,)


class BalanceContractRuleViolation(BaseBalanceError):
    MESSAGE = 'BALANCE_CONTRACT_RULE_VIOLATION'
    CODES = ('CONTRACT_RULE_VIOLATION',)


class BalanceDataError(BaseBalanceError):
    MESSAGE = 'BALANCE_DATA_ERROR'


class BalanceWrongEmailError(BalanceDataError):
    MESSAGE = 'BALANCE_WRONG_EMAIL'
    CODES = ('WRONG_EMAIL',)


class BalanceInvalidParam(BalanceDataError):
    MESSAGE = 'BALANCE_INVALID_PARAM'
    CODES = ('INVALID_PARAM',)


class BalanceMissingMandatoryPersonField(BalanceInvalidParam):
    MESSAGE = 'BALANCE_MISSING_MANDATORY_PERSON_FIELD'
    CODES = ('MISSING_MANDATORY_PERSON_FIELD',)


class BalanceWrongAccount(BalanceDataError):
    MESSAGE = 'BALANCE_INVALID_ACCOUNT'
    CODES = ('WRONG_ACCOUNT',)


class BalanceWrongBIK(BalanceDataError):
    MESSAGE = 'BALANCE_INVALID_BIK'
    CODES = ('WRONG_BIK',)


class BalanceInvalidINN(BalanceDataError):
    MESSAGE = 'BALANCE_INVALID_INN'
    CODES = ('INVALID_INN',)


class BalanceDenyChangeINN(BalanceDataError):
    MESSAGE = 'BALANCE_DENY_CHANGE_INN'
    CODES = ('CHANGING_INN_IS_PROHIBITED',)


class BalancePassportNotFound(BalanceDataError):
    MESSAGE = 'BALANCE_USER_DOES_NOT_EXIST'
    CODES = ('PASSPORT_NOT_FOUND',)


class BalanceContractNotFound(BalanceDataError):
    MESSAGE = 'BALANCE_CONTRACT_NOT_FOUND'
    CODES = ('CONTRACT_NOT_FOUND',)
